/**
* Copyright (c) 2021-2025 Kedron Holdings LLC, All Rights Reserved.
* This file is not part of Reactor, but a free add-on for it. This file only is offered under the MIT License.
* See https://mit-license.org for license text/details.
*/

const version = 25099;

const Controller = require("server/lib/Controller").requires( 24273 );
const Capabilities = require("server/lib/Capabilities");

const Logger = require("server/lib/Logger");
const util = require("server/lib/util");

const net = require("net");

Logger.getLogger( 'EnvisalinkController', Logger.getLogger( 'Controller' ) )
    .always( "Module EnvisalinkController v%1", version );

const DEFAULT_PORT = 4025;

const POLL_INTERVAL = 900000;  // 15 minutes
const DATA_TIMEOUT = 30000;    // 30 secs (panel data every 10 seconds is normal)

const ICONBITS = {
    0x8000: "ARMED STAY",
    0x4000: "LOW BATTERY",
    0x2000: "FIRE",
    0x1000: "READY",
    0x0800: null,
    0x0400: null,
    0x0200: "SYSTEM TROUBLE",
    0x0100: "ALARM (FIRE ZONE)",
    0x0080: "ARMED (INSTANT)",
    0x0040: null,
    0x0020: "CHIME",
    0x0010: "BYPASS",
    0x0008: "AC PRESENT",
    0x0004: "ARMED AWAY",
    0x0002: "ALARM IN MEMORY",
    0x0001: "ALARM"
};

const STATUSMASK = 0xF3BF;  // masks off unknown bits
const TROUBLEBITS = 0x4202; // masks off trouble bits

const modemap = { "disarmed": 1, "away": 2, "stay": 3, "max": 4, "instant": 7 };

//                   00       01          02          03        04      05       06       07    08    09    10
const partModes = [ null, "disarmed", "disarmed", "disarmed", "stay", "away", "instant", null, null, null, 'max' ]

module.exports = class EnvisalinkController extends Controller {
    constructor( struct, id, config ) {
        super( struct, id, config );

        if ( util.coalesce( this.config.panel, "honeywell" ) !== "honeywell" ) {
            throw new Error("Unsupported panel type");
        }

        this.num_fail = 0;
        this.stopping = false;
        this.socket = false;
        this.connected = false;
        this.retries = 0;

        this.troubleSince = new Array(16).fill(0);

        this.dataTimer = this.getTimer( this.id + "-socket", this._recycle.bind( this ) );
        this.pollTimer = this.getTimer( this.id + "-poller", this._tpi_poll.bind( this ) );

        this.host = util.coalesce( this.config.host, "127.0.0.1" );
        this.port = util.coalesce( parseInt( this.config.port ), DEFAULT_PORT );
        this.partitions = Math.max( 0, util.coalesce( parseInt( this.config.partitions ), 1 ) );
        this.zones = Math.max( 0, Math.min( 128, util.coalesce( parseInt( this.config.zones ), 64 ) ) );

        this.trouble_sustain = util.coalesce( parseInt( this.config.trouble_sustain ), 30 ) * 1000;

        if ( ! Capabilities.getCapability( 'x_envisalink' ) ) {
            Capabilities.loadCapabilityData( {
                "x_envisalink": {
                    "attributes": {
                        "status": {
                            "type": "string"
                        },
                        "status_mask": {
                            "type": "ui2"
                        },
                        "zones_faulted": {
                        },
                        "reconnects": {
                            "type": "int"
                        }
                    },
                    "actions": {
                        "send_command": {
                            "arguments": {
                                "data": {
                                    "type": "string"
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    /** Start the controller.
     */
    async start() {
        const eset = new Set( Object.keys( this.getEntities() ) );
        eset.delete( "system" );
        eset.delete( "controller_all" );

        let e = this.getSystemEntity();
        e.markDead( false );
        e.deferNotifies( true );
        e.extendCapabilities([
            'sys_system',
            'x_envisalink',
            'security_mode',
            'battery_maintenance',
            'battery_power',
            'power_source'
        ]);
        e.refreshCapabilities();
        e.setPrimaryAttribute( "security_mode.mode" );
        e.setAttribute( 'x_envisalink.reconnects', 0 );
        if ( null !== this.config.function_keys && "object" === typeof this.config.function_keys && ! Array.isArray( this.config.function_keys ) ) {
            for ( let [k,v] of Object.entries(this.config.function_keys) ) {
                const action = `x_envisalink.key_${v}`;
                this.log.info( "%1 registering action %2 for function key %3", this, action, k );
                e.registerAction( action, ( entity, action /* , params */ ) => {
                    this.log.info( "%1 performing action %2: sending function key %3", this, action, k );
                    return this.action_x_envisalink_send_command( entity, { data: String(k) } );  /* Returns Promise */
                });
            }
        }
        e.deferNotifies( false );

        e = this.getEntity( 'Entity', 'in_alarm' );
        e.markDead( false );
        e.deferNotifies( true );
        e.extendCapability( 'binary_sensor' );
        e.refreshCapabilities();
        e.setPrimaryAttribute( 'binary_sensor.state' );
        e.deferNotifies( false );
        eset.delete( "in_alarm" );

        e = this.getEntity( 'Entity', 'trouble' );
        e.markDead( false );
        e.deferNotifies( true );
        e.extendCapability( 'binary_sensor' );
        e.refreshCapabilities();
        e.setPrimaryAttribute( 'binary_sensor.state' );
        e.deferNotifies( false );
        eset.delete( "trouble" );

        e = this.getEntity( 'Entity', 'panel_message' );
        e.markDead( false );
        e.deferNotifies( true );
        e.extendCapability( 'string_sensor' );
        e.refreshCapabilities();
        e.setPrimaryAttribute( 'string_sensor.value' );
        e.deferNotifies( false );
        eset.delete( "panel_message" );

        for ( let k=1; k<=this.zones; ++k ) {
            const eid = `zone${("0"+String(k)).substr(-2)}`;
            e = this.getEntity( 'Entity', eid );
            e.markDead( false );
            e.deferNotifies( true );
            e.extendCapabilities(['binary_sensor','string_sensor','security_zone']);
            e.refreshCapabilities();
            e.setPrimaryAttribute( 'binary_sensor.state' );
            e.deferNotifies( false );
            eset.delete( eid );
        }

        for ( let k=1; k<=this.partitions; ++k ) {
            const eid = `part${("0"+String(k)).substr(-2)}`;
            e = this.getEntity( 'Entity', eid );
            e.markDead( false );
            e.deferNotifies( true );
            e.extendCapabilities(['binary_sensor','value_sensor','security_mode']);
            e.refreshCapabilities();
            e.setPrimaryAttribute( 'binary_sensor.state' );
            e.registerAction( "security_mode.set_mode", false );  // not yet implemented
            e.deferNotifies( false );
            eset.delete( eid );
        }

        eset.forEach( eid => {
            const e = this.findEntity( eid );
            if ( e ) {
                e.markDead( true );
            }
        });

        this.startDelay( 10 );
        return this;
    }

    async stop() {
        this.stopping = true;
        this.stopTimer();
        this.dataTimer.cancel();
        this.pollTimer.cancel();
        // ??? we don't dispose of the above timers, but we should -- how?

        this._disconnect();

        await super.stop();
    }

    run() {
        if ( ! this.socket ) {
            this._connect();
        }
    }

    _connect() {
        const self = this;

        this._disconnect();

        const socket = new net.Socket({ readable: true, writable: true });
        socket.setTimeout( 30000 );

        socket.on( 'close', (haderror) => {
            self.log.info( "%1 socket closing, error=%2", self, haderror );
            self.connected = false;
            self.socket = false;
            self.stopTimer();
            self.offline();
            if ( ! this.stopping ) {
                self.startDelay( Math.min( 30000, 10 + 1000 * self.retries ) );
            }
        });
        socket.on( 'connect', () => {
            self.log.debug( 5, "%1 socket connected", self );
        });
        socket.on( 'ready', () => {
            self.log.notice( "%1 connected to %2:%3; waiting for data", self, this.host, this.port );
            self.connected = true;
            self.online();
            self.retries = 0;
            const se = this.getSystemEntity();
            se.setAttribute( 'x_envisalink.reconnects', util.coalesce( se.getAttribute( 'x_envisalink.reconnects' ), 0 ) + 1 );
            this.dataTimer.delayms( DATA_TIMEOUT );
            this.pollTimer.cancel();  // restarts with login
        });
        socket.on( 'data', (data) => {
            try {
                self._handleSocketData( data );
            } catch ( err ) {
                this.log.exception( err );
            }
        });
        socket.on( 'error', (err) => {
            self.log.err( "%1 socket error: %2", self, err );
        });
        socket.on( 'timeout', (err) => {
            self.log.warn("%1 socket timeout, recycling", self);
            self._recycle();
        });

        self.log.info( "%1 connecting to %2:%3", self, this.host, this.port );
        ++self.retries;
        socket.connect({
            host: this.host,
            port: this.port,
            noDelay: true
        });

        self.socket = socket;
    }

    _disconnect() {
        this.dataTimer.cancel();
        // don't reset poll timer, because this may not have anything to do with connection state, it's global to device

        if ( this.socket ) {
            this.socket.end();
            this.socket.unref();
            this.socket.destroy();
        }
        this.socket = false;
        this.connected = false;
    }

    _recycle() {
        this._disconnect();
        if ( ! this.stopping ) {
            this.stopTimer();
            this.startDelay(10);
        }
    }

    _handleSocketData( b ) {
        this.log.debug( 5, "%1 received socket data: %2", this, b );
        let s = b.toString( 'utf8' ).trim();
        if ( s.startsWith( 'Login:' ) ) {
            this.socket.write( this.config.password || "user" );
            this.socket.write( "\r" );
            return;
        }
        this.dataTimer.delayms( DATA_TIMEOUT );  // extend the dead connection timer
        // From here, buffer may contain multiple messages/message types
        while ( s.length ) {
            if ( s.startsWith( "\r" ) || s.startsWith( "\n" ) ) {
                s = s.substr( 1 );
            } else if ( s.startsWith( '%' ) ) {
                let k = s.indexOf( '$' );
                if ( k < 0 ) {
                    this.log.warn( "%1 dropping malformed message: %2", this, s );
                    break;
                }
                const cmd = s.substring( 1, k );
                s = s.substr( k+1 );
                this.log.debug(5, "%1 handling command %2", this, cmd );
                const parts = cmd.split( ',' );
                switch ( parseInt( parts[0], 16 ) ) {
                    case 0:
                        this._keypad_update( parts );
                        break;
                    case 1:
                        this._zone_state_change( parts );
                        break;
                    case 2:
                        this._partition_state_change( parts );
                        break;
                    case 3:
                        this._cid_event( parts );
                        break;
                    default:
                        // nada
                }
            } else if ( s.startsWith( "^" ) ) {
                let k = s.indexOf( '$' );
                if ( k < 0 ) {
                    this.log.warn( "%1 dropping malformed message: %2", this, s );
                    break;
                }
                const cmd = s.substring( 1, k );
                s = s.substr( k+1 );
                this.log.debug( 5, "%1 command response: %2", this, cmd );
            } else if ( s.startsWith( "OK" ) ) {
                // Ack for login, quietly ignore
                s = s.substr( 2 );
                this.pollTimer.delayms( 1000 );
            } else if ( s.startsWith( "FAILED" ) ) {
                this.log.err( "%1 password failed for connection to %2", this, this.config.host );
                // let it time out
            } else {
                this.log.notice( "%1 unrecognized panel data: %2", this, s );
                s = s.substr( 1 );
            }
        }
    }

    _keypad_update( parts ) {
        const now = Date.now();
        const partition = parseInt( parts[1] );
        let status = parseInt( parts[2], 16 ) & STATUSMASK;
        const userzone = parseInt( parts[3] );
        const beep = parseInt( parts[4] );
        const message = parts[5].trim();

        /** Separate the status bits from the trouble bits. The trouble bits tend to toggle when the alpha display
         *  alternates between the status (armed/disarmed state) and the trouble message. We're dampening that here.
         *  We keep a timer for each trouble bit, updating it to current time whenever the bit is set. When the bit
         *  is reset/zero, we check the timer to see if it's been more than 30 secs (default) since it was last on.
         *  If so, we allow it to turn off in the status word; otherwise, we keep it on.
         */
        let trouble = status & TROUBLEBITS;
        this.log.debug( 5, "%1 trouble=%2", this, trouble.toString(16) );
        if ( trouble || this.troubleSince.find( el => el > 0 ) ) {
            status = status & ~TROUBLEBITS;  // all trouble bits off
            for ( let k=0; k<16; ++k ) {
                if ( trouble & (1<<k) ) {
                    // Trouble bit is set. Update time.
                    this.troubleSince[k] = now;
                    status |= (1<<k);  // Put it back on status
                    this.log.debug( 5, "%1 trouble bit %2 is ON, updating time", this, k);
                } else if ( this.troubleSince[k] ) {
                    // Bit is off, but was on recently (we have a timestamp). Check expiration.
                    if ( now >= ( this.troubleSince[k] + this.trouble_sustain ) ) {
                        // expired, clear timer.
                        this.log.debug( 5, "%1 trouble bit %2 is OFF, timer expired", this, k);
                        this.troubleSince[k] = 0;
                    } else {
                        // not expired; force bit back on in status to dampen entity changes
                        this.log.debug( 5, "%1 trouble bit %2 is OFF, not yet expired", this, k);
                        status |= (1<<k);
                    }
                }
            }
        }

        let s = [];
        for ( let bit in ICONBITS ) {
            if ( ( bit & status ) && ICONBITS[bit] ) {
                s.push( ICONBITS[bit] );
            }
        }

        let e = this.getSystemEntity();
        e.deferNotifies( true );
        try {
            if ( this.config.log_panel && message !== e.getAttribute( 'string_sensor.value' ) ) {
                this.log.notice( "%1 message %2", this, message );
            }

            e.setAttribute( 'power_source.source', ( status & 0x0008 ) ? 'ac' : 'battery' );
            let k = ( status & 0x4000 ) ? 0 : 1;
            if ( e.getAttribute( 'battery_power.level' ) !== k ) {
                e.setAttribute( 'battery_power.level', k );
                e.setAttribute( 'battery_power.since', Date.now() );
            }

            e.setAttributes({
                charging: (status & 0x0008) !== 0,  // AC power on, assumed to be charging
                rechargeable: true,
                replace: null,
                state: (status & 0x4000) ? "maintenance" : "normal"
            }, "battery_maintenance");

            let mode = "unknown";
            if ( status & 0x8000 ) {
                mode = "stay";
            } else if ( status & 0x0080 ) {
                mode = "instant";
            } else if ( status & 0x0004 ) {
                mode = "away";
            } else {
                mode = "disarmed";
            }
            e.setAttributes({
                mode: mode,
                ready: ( status & 0x1000 ) !== 0
            }, "security_mode");

            if ( false !== this.config.log_status && status !== e.getAttribute( 'x_envisalink.status_mask' ) ) {
                this.log.notice( "%1 status: %2 from: %3", this, s.join(', '), parts );
            }
            e.setAttributes({
                status_mask: status,
                status: s
            }, "x_envisalink");
            e.setAttributes({
                numeric: util.coalesce( userzone ),
                beep: util.coalesce( beep )
            }, "x_envisalink", { mark_dirty: false });  // don't mark dirty for these

        } finally {
            e.deferNotifies( false );
        }

        e = this.findEntity( 'panel_message' );
        e.setAttribute( 'string_sensor.value', message );

        e = this.findEntity( 'in_alarm' );
        e.setAttribute( 'binary_sensor.state', ( status & 0x0101 ) !== 0 );

        e = this.findEntity( 'trouble' );
        e.setAttribute( 'binary_sensor.state', ( status & TROUBLEBITS ) !== 0 );

        // Based on zones faulting, "learn" the zone names
        if ( message.startsWith( "FAULT " ) ) {
            // FAULT ZZ NAME
            const zone = message.substring( 6, 8 );
            const eid = `zone${zone}`;
            e = this.findEntity( eid );
            if ( e && e.getName() === eid ) {  // Only reset name if default name (entity ID) is in effect.
                const nom = message.substring( 9 ).trim();
                e.setName( nom );
            }
        }
/*
        if ( false && message.startsWith( "LOBAT " ) ) {
            // Low battery on zone -- set battery_maintenance.state to maintenence. ??? How would we recover?
            const zone = parseInt( message.substring( 6, 8 ) );
            const eid = `zone${zone}`;
            e = this.findEntity( eid );
            if ( e ) {
                e.deferNotifies( true );
                e.extendCapability( 'battery_maintenance' );
                e.refreshCapability( 'battery_maintenance' );
                e.setAttribute( 'battery_maintenance.state', 'maintenance' );
                e.deferNotifies( false );
            }
        }
*/
    }

    _zone_state_change( parts ) {
        // 8 or 16 byte hex string for zone states. First byte is zones 1-8, second is 9-16, etc.
        // 0100000000000080 means zone 1 and 64 are open/faulted (8 bytes EVL3, 16 for EVL4)
        let s = parts[1];
        let zone = 1;
        let faults = [];
        while ( s.length > 1 ) {
            if ( zone > this.zones ) {
                break;
            }
            let n = parseInt( s.substring( 0, 2 ), 16 );
            s = s.substring( 2 );
            for ( let i=0; i<8; ++i ) {
                if ( zone > this.zones ) {
                    break;
                }
                this.log.debug(5, "%1 checking zone %2 byte %3 bit %4", this, zone, n, i);
                const e = this.findEntity( `zone${('0'+String(zone)).substr(-2)}` );
                e.deferNotifies( true );
                try {
                    const state = ( n & ( 1<<i ) ) !== 0;
                    if ( state ) {
                        faults.push( e.getID() );
                    }
                    if ( e.getAttribute( 'binary_sensor.state' ) !== state ) {
                        if ( false !== this.config.log_status ) {
                            this.log.notice( `%1 ${state ? "fault" : "restored"} zone ${zone} %2`, this, e.getName() );
                        }
                        e.setAttribute( 'binary_sensor.state', state );
                    }
                    const zs = state ? "faulted" : "normal";
                    if ( e.getAttribute( 'security_zone.state' ) !== zs ) {
                        e.setAttribute( 'security_zone.state', zs );
                    }
                } catch ( err ) {
                    this.log.exception( err );
                } finally {
                    e.deferNotifies( false );
                }

                ++zone;
            }
        }

        this.getSystemEntity().setAttribute( 'x_envisalink.zones_faulted', faults );
    }

    _partition_state_change( parts ) {
        // Partition states, like zone states but bytes, not bits.
        let s = parts[1];
        let part = 1;
        while ( s.length > 1 ) {
            if ( part > this.partitions ) {
                break;
            }
            let n = parseInt( s.substring( 0, 2 ), 16 );
            s = s.substring( 2 );
            this.log.debug(5, "%1 checking partition %2 byte %3", this, part, n);
            const e = this.findEntity( `part${('0'+String(part)).substr(-2)}` );
            e.deferNotifies( true );
            try {
                e.setAttribute( 'binary_sensor.state', 8 === n );
                e.setAttribute( 'value_sensor.value', n );
                e.setAttributes({
                    ready: n < 3,
                    mode: util.coalesce( partModes[ n ], String( n ) )
                }, 'security_mode' );
            } catch ( err ) {
                this.log.exception( err );
            } finally {
                e.deferNotifies( false );
            }

            ++part;
        }
    }

    _cid_event( parts ) {
    }

    _tpi_poll() {
        if ( this.connected ) {
            this.log.debug(5, "%1 sending periodic TPI poll");
            this.socket.write( "^00,$" );
            this.pollTimer.delayms( POLL_INTERVAL );
        }
    }

    action_security_mode_set_mode( entity, params ) {
        let code = ( params?.code || "" ).trim();
        if ( util.isEmpty( code ) ) {
            code = this.config.usercode || "";
        }
        if ( util.isEmpty( code ) ) {
            return Promise.reject( new Error( "Missing user code" ) );
        }

        let s = code + ( modemap[ params?.mode ] || "*0" );
        this.log.info( "%1 setting security mode to %2", this, params?.mode );
        this.log.debug(5, "%1 action security_mode.set_mode sending data: %2", this, s);
        this.socket.write( s );
        return Promise.resolve();
    }

    action_x_envisalink_send_command( entity, params ) {
        const self = this;
        return new Promise( (resolve, reject) => {
            if ( "string" === typeof params.data && params.data.length > 0 ) {
                self.log.info( "%1 sending %2", self, params.data );
                self.socket.write( params.data );
                resolve();
            } else {
                reject(new Error("Invalid data"));
            }
        });
    }

    action_sys_system_restart( /* entity, params */ ) {
        const self = this;
        return new Promise( (resolve) => {
            self.log.notice( "%1 action sys_system.restart: recycling connection", self );
            self._recycle();
            resolve();
        });
    }
};
