/**
* Copyright (c) 2021-2025 Kedron Holdings LLC, All Rights Reserved.
* This file is not part of Reactor, but a free add-on for it. This file only is offered under the MIT License.
* See https://mit-license.org for license text/details.
*/

/** NOTA BENE -- The Lync6/12+GW-SL1 gateway and accompanying documentation have a lot of bugs and inconsistencies.
 *  There are commands that are reported to affect zones individually that affect all zones (e.g. set default names).
 *  There are commands documented with value ranges that the gateway doesn't accept (e.g. set bass/treble/balance).
 *  Many values do not "stick" unless the zone is powered on (e.g. muting, DND), and are reset when the zone is
 *      turned off.
 *  Some values can be changed, but don't cause a zone status update from the gateway (e.g. bass, treble, balance).
 *      This is worked around by setting the muting state to its current state (i.e. muting no-op), which causes
 *      a zone status update packet.
 *  Querying source name on a zone requires that you send the source number zero-based (i.e. send 0 for source 1);
 *      it's one-based everywhere else. Solid work, guys.
 */

const version = 25251;

const Controller = require("server/lib/Controller").requires( 24273 );
const Capabilities = require("server/lib/Capabilities");

const Logger = require("server/lib/Logger");
const util = require("server/lib/util");

const net = require("net");

Logger.getLogger( 'HTDLyncController', Logger.getLogger( 'Controller' ) )
    .always( "Module HTDLyncController v%1", version );

const DEFAULT_PORT = 10006;

const DATA_TIMEOUT = 600000; // 10 minutes (normally get info packets every 5 minutes)

// Expected length of various messages
const msg_len = {
    0x05: 9,        // Zone Internal Status
    0x06: 9,        // Audio and keypad exist channel
    0x09: 1,        // MP3 Play End
    0x0c: 13,       // Zone Source Name
    0x0d: 13,       // Zone Name
    0x0e: 13,       // Zone source name response
    0x11: -1,       // MP3 file name (variable)
    0x12: -1,       // MP3 artist name (variable)
    0x13: -1,       // MP3 ON
    0x14: -1,       // MP3 OFF
    0x1b: 9,        // Error Status
};

const zoneSources = [];

module.exports = class HTDLyncController extends Controller {
    constructor( struct, id, config ) {
        super( struct, id, config );

        if ( util.coalesce( this.config.panel, "honeywell" ) !== "honeywell" ) {
            throw new Error("Unsupported panel type");
        }

        this.num_fail = 0;
        this.stopping = false;
        this.socket = false;
        this.connected = false;
        this.retries = 0;

        this.dataTimer = this.getTimer( this.id + "-socket", this._recycle.bind( this ) );

        this.host = util.coalesce( this.config.host, "127.0.0.1" );
        this.port = util.coalesce( parseInt( this.config.port ), DEFAULT_PORT );

        this.zones = util.coalesce( parseInt( this.config.zones), 12 );

        this.buffer = Buffer.from( [] );

        if ( ! Capabilities.getCapability( 'x_htdlync' ) ) {
            Capabilities.loadCapabilityData( {
                "x_htdlync": {
                    "attributes": {
                        "reconnects": {
                            "type": "int",
                            "default": 0
                        }
                    },
                    "actions": {
                        "party_mode": {
                            "arguments": {
                                "source": {
                                    "type": "string",
                                    "sort": 1
                                },
                                "zone": {
                                    "type": "int",
                                    "optional": true,
                                    "sort": 2
                                }
                            }
                        },
                        "set_all_volume": {
                            "arguments": {
                                "value": {
                                    "type": "real",
                                    "min": 0,
                                    "max": 1,
                                }
                            }
                        },
                        "set_all_volume_db": {
                            "arguments": {
                                "db": {
                                    "type": "int",
                                    "min": -61,
                                    "max": 0,
                                }
                            }
                        },
                        "set_all_source_name": {
                            "arguments": {
                                "source": {
                                    "type": "int",
                                    "min": 1,
                                    "max": 18,
                                    "sort": 1
                                },
                                "name": {
                                    "type": "string",
                                    "sort": 2
                                }
                            }
                        },
                        "refresh": {
                            "arguments": {}
                        }
                    }
                },
                "x_htdlync_zone": {
                    "attributes": {
                        "zoneid": {},
                        "name": {},
                        "keypad": {},
                        "dnd": {},
                        "volume": {},
                        "bass": {},
                        "treble": {},
                        "balance": {},
                        "input": {},
                        "party_mode": {}
                    },
                    "actions": {
                        "set_zone_name": {
                            "arguments": {
                                "name": {
                                    "type": "string",
                                    "sort": 1
                                }
                            }
                        },
                        "set_source_name": {
                            "arguments": {
                                "source": {
                                    "type": "int",
                                    "min": 1,
                                    "max": 18,
                                    "sort": 1
                                },
                                "name": {
                                    "type": "string",
                                    "sort": 2
                                }
                            }
                        },
                        "set_balance": {
                            "arguments": {
                                "balance": {
                                    "type": "int",
                                    "min": -18,
                                    "max": 18
                                }
                            }
                        },
                        "set_treble": {
                            "arguments": {
                                "treble": {
                                    "type": "int",
                                    "min": -10,
                                    "max": 10
                                }
                            }
                        },
                        "set_bass": {
                            "arguments": {
                                "bass": {
                                    "type": "int",
                                    "min": -10,
                                    "max": 10
                                }
                            }
                        },
                        "set_dnd": {
                            "arguments": {
                                "dnd": {
                                    "type": "bool"
                                }
                            }
                        },
                        "set_audio_default": {
                            "arguments": {}
                        },
                        "set_names_default": {
                            "arguments": {}
                        }
                    }
                }
            });
        }
    }

    /** Start the controller.
     */
    async start() {
        const eset = new Set( Object.keys( this.getEntities() ) );
        eset.delete( "system" );
        eset.delete( "controller_all" );

        let e = this.getSystemEntity();
        e.markDead( false );
        e.deferNotifies( true );
        e.extendCapabilities([
            'sys_system',
            'x_htdlync',
            'av_transport',
            'media_navigation',
            'av_repeat',
            'power_switch',
            'toggle'
        ]);
        if ( e._ctrl_version !== version ) {
            e.refreshCapabilities();
            e._ctrl_version = version;
        }
        e.setAttribute( 'x_htdlync.reconnects', 0 );
        e.deferNotifies( false );

        this.startDelay( 10 );
        return this;
    }

    async stop() {
        this.stopping = true;
        this.stopTimer();
        this.dataTimer.cancel();
        // ??? we don't dispose of the above timers, but we should -- how?

        this._disconnect();

        await super.stop();
    }

    run() {
        if ( ! this.socket ) {
            this._connect();
        }
    }

    _connect() {
        const self = this;

        this._disconnect();

        const socket = new net.Socket({ readable: true, writable: true });
        //socket.setTimeout( 30000 );

        socket.on( 'close', (haderror) => {
            self.log.info( "%1 socket closing %2", self, haderror );
            self.connected = false;
            self.socket = false;
            self.stopTimer();
            if ( ! this.stopping ) {
                self.startDelay( Math.min( 60000, 10 + 1000 * self.retries ) );
            }
        });
        socket.on( 'connect', () => {
            self.log.debug( 5, "%1 socket connected", self );
        });
        socket.on( 'ready', () => {
            self.log.notice( "%1 connected to %2:%3; querying full status", self, this.host, this.port );
            self.connected = true;
            self.online();
            self.retries = 0;
            const se = this.getSystemEntity();
            se.setAttribute( 'x_htdlync.reconnects', util.coalesce( se.getAttribute( 'x_htdlync.reconnects' ), 0 ) + 1 );
            this.dataTimer.delayms( DATA_TIMEOUT );
            self._sendData( Buffer.from( [0x02,0x00,0x00,0x19,0xff] ) );  // Echo ON
            self._sendData( Buffer.from( [0x02,0x00,0x01,0x0c,0x00] ) );  // Query ALL
        });
        socket.on( 'data', (data) => {
            try {
                self.dataTimer.delayms( DATA_TIMEOUT );
                self._handleSocketData( data );
            } catch ( err ) {
                this.log.exception( err );
            }
        });
        socket.on( 'error', (err) => {
            self.log.err( "%1 socket error: %2", self, err );
        });
        socket.on( 'timeout', (err) => {
            self.log.warn("%1 socket timeout, recycling", self);
            self._recycle();
        });

        self.log.info( "%1 connecting to %2:%3", self, this.host, this.port );
        if ( ++self.retries >= 3 ) {
            self.offline();
        }
        socket.connect({
            host: this.host,
            port: this.port,
            noDelay: true
        });

        self.socket = socket;
    }

    _disconnect() {
        this.dataTimer.cancel();
        // don't reset poll timer, because this may not have anything to do with connection state, it's global to device

        if ( this.socket ) {
            this.socket.end();
            this.socket.unref();
            this.socket.destroy();
        }
        this.socket = false;
        this.connected = false;
    }

    _recycle() {
        this._disconnect();
        if ( ! this.stopping ) {
            this.stopTimer();
            this.startDelay( 1000 );
        }
    }

    // Send data (array or Buffer), appending checksum byte.
    _sendData( b ) {
        const bx = Buffer.allocUnsafe( b.length + 1 );
        let checksum = 0;
        for ( let ix=0; ix<b.length; ++ix ) {
            bx[ ix ] = b[ ix ];
            checksum = ( checksum + b[ ix ] ) & 0xff;
        }
        bx[ b.length ] = checksum;
        this.log.debug( 5, "%1 writing %2", this, bx.toString("hex") );
        this.socket.write( bx );
    }

    /** As it turns out, setting some values like base, treble, and balance don't cause a zone status
     *  update from the gateway. But, we can work around that by setting the muting state to its current
     *  state (i.e. a muting no-op), which causes a zone status message to be sent.
     */
    async _zone_refresh_workaround( entity ) {
        const muting = entity.getAttribute( "muting.state" );
        await this.action_muting_set( entity, { state: muting } );
    }

    _handleSocketData( b ) {
        this.log.debug( 5, "%1 received socket data, %2 bytes; buffer currently has %3", this, b.length,
            this.buffer.length );

        // Append received data to the buffer.
        this.buffer = Buffer.concat( [ this.buffer, b ] );

        /**
         *  A full message starts with a four byte header: 0x02 0x00 zone command
         *  Each message will also have some number of data bytes, dependent on the command. At this time, there are
         *  no commands without data (so data length is always at least 1, FOR NOW).
         *  always includes at least one data byte.
         *  The final byte of the message is the checksum, a single 8-bit sum of all preceding bytes.
         *
         *  The data lengths are fixed for most commands, but a few have variable-length data. These commands have
         *  a -1 length in the msg_len table.
         *
         *  To process any message, we need at least 5 bytes (assuming data length could someday be 0 in some new
         *  message to be added to the protocol later).
         */
        while ( this.buffer.length > 4 ) {
            this.log.debug( 6, "%1 processing buffer with %2 bytes", this, this.buffer.length );
            if ( this.buffer[ 0 ] !== 0x02 || this.buffer[ 1 ] !== 0x00 ) {
                // Try to resync by finding a 0x02 0x00 pair
                var sync = function( b ) {
                    let ix = 1;
                    const last = b.length - 1;
                    while ( ix < last ) {
                        if ( b[ix] === 0x02 && b[ix+1] === 0x00 ) {
                            return ix;
                        }
                        ++ix;
                    }
                    return ix;
                };
                const n = sync( this.buffer );
                this.log.warn( "%1 ignoring data: %2", this,
                    this.buffer.slice( 0, n ).toString( "hex" ).replace( /(..)/g, "\$1 " ) );
                this.buffer = this.buffer.slice( n );
                continue;
            }

            // Header is first 4 bytes, checksum is last byte. Message length is type-dependent.
            const typ = this.buffer[ 3 ];
            const datalen = msg_len[ typ ];
            if ( "undefined" !== typeof datalen ) {
                this.log.debug( 6, "%1 found type %2 data length %3", this, typ, datalen );
                if ( datalen >= 0 ) {
                    const need = 5 + datalen;  // data length + 4 bytes header + 1 byte checksum
                    if ( this.buffer.length < need ) {
                        this.log.debug( 5, "%1 packet incomplete; waiting for more data", this );
                        break;  // wait for more data.
                    }
                    const packet = this.buffer.slice( 0, need );
                    this.buffer = this.buffer.slice( need );
                    this._handlePacket( packet );
                } else {
                    // Variable-length data. Seek a zero byte after the header.
                    const ix = this.buffer.indexOf( 0x00, 4 );
                    if ( ix < 0 ) {
                        // Didn't find a zero byte, maybe more data coming?
                        this.log.debug( 5, "%1 packet incomplete; waiting for more data", this );
                        break;
                    }
                    const packet = this.buffer.slice( 0, ix + 2 );  // include checksum
                    this.buffer = this.buffer.slice( ix + 2 );
                    this._handlePacket( packet );
                }
            } else {
                this.log.warn( "%1 unknown message type 0x%2", this, typ.toString(16) );
                // Force resync.
                this.buffer = this.buffer.slice( 1 );
            }
        }
    }

    _handlePacket( packet ) {
        this.log.debug( 6, "%1 handling packet %2", this, packet.toString("hex").replace(/(..)/g, "\$1 ") );
        let ck = 0;
        for ( let ix=0; ix<packet.length-1; ++ix ) {
            ck += packet[ ix ];
        }
        if ( ( ck & 0xff ) !== packet[ packet.length-1 ] ) {
            this.log.err( "%1 packet checksum error: %2", this, packet.toString("hex").replace(/(..)/g, "\$1 ") );
            return;
        }

        const typ = packet[ 3 ];
        const zoneid = packet[ 2 ];
        const data = packet.slice( 4, packet.length - 1 );
        if ( 0x06 === typ && 0x00 === zoneid ) {
            this.log.debug( 5, "%1 handling audio/keypad broadcast", this );
            var divid = 1, byt = 1;  // data[0] is ignored
            for ( let zone=1; zone <= this.zones; ++zone ) {
                const stz = data[ byt ] & divid;
                const stk = data[ byt + 1 ] & divid;

                if ( stz ) {
                    const eid = `zone${zone}`;
                    let e = this.findEntity( eid );
                    if ( !e ) {
                        // Create new.
                        e = this.getEntity( 'Entity', eid );
                        e.setName( eid );
                        e.extendCapabilities([
                            'x_htdlync_zone',
                            'power_switch',
                            'toggle',
                            'volume',
                            'muting',
                            'media_source'
                        ]);
                    } else if ( e._ctrl_version !== version ) {
                        e.refreshCapabilities();
                        e._ctrl_version = version;
                    }
                    e.setPrimaryAttribute( 'power_switch.state' );
                    e.setAttributes({
                        'x_htdlync_zone.zoneid': zone,
                        'x_htdlync_zone.keypad': stk !== 0
                    });
                    e.markDead( false );
                    e.deferNotifies( false );

                    zoneSources[ zone ] = [];
                    for ( let source = 1; source <= 12; ++source ) {
                        const nom = e.getAttribute( `x_htdlync_zone.source${source}` );
                        if ( nom ) {
                            zoneSources[ zone ][ source ] = nom;
                        }
                    }
                    this.log.debug( 5, "%1 reloaded zoneSources for %2: %3", this, zone, zoneSources[ zone ] );
                }

                // Next!
                divid <<= 1;
                if ( 8 === zone ) {
                    byt += 2;
                    divid = 1;
                }
            }
        } else if ( 0x05 === typ ) {
            // Zone status
            this.log.debug( 5, "%1 handling zone %2 status", this, zoneid );
            const eid = `zone${zoneid}`;
            const e = this.findEntity( eid );
            if ( e ) {
                let power = data[0] & 0x01; // power on/off
                let mute = data[0] & 0x02;  // mute on/off
                let DND = data[0] & 0x04;   // DND on/off
                // data[1] & 0x01 -- All ON
                // data[1] & 0x02 -- All OFF
                // data[1] & 0x04 -- Party Mode
                // data[2] & 0x10 -- MP3 Repeat Loop
                // data[3] & 0x10 -- MP3 Repeat Loop (Party Mode)
                // data[4] -- input port 0-12
                let volume = data[5] << 24 >> 24;
                let treble = data[6] << 24 >> 24;
                let bass = data[7] << 24 >> 24;
                let balance = data[8] << 24 >> 24;

                try {
                    e.setAttributes({
                        'power_switch.state': power !== 0,
                        'muting.state': mute !== 0,
                        'volume.level': Math.floor( ( 61 + volume ) * 100 / 61 ) / 100,
                        'media_source.selected': e.getAttribute( `x_htdlync_zone.source${data[4]+1}` ) || "unknown",
                        'x_htdlync_zone.dnd': DND !== 0,
                        'x_htdlync_zone.party_mode': ( data[1] & 0x04 ) !== 0,
                        'x_htdlync_zone.volume': volume,
                        'x_htdlync_zone.treble': treble,
                        'x_htdlync_zone.bass': bass,
                        'x_htdlync_zone.balance': balance,
                        'x_htdlync_zone.input': data[4]
                    });
                } catch ( err ) {
                    this.log.err( "%1 can't set attributes: %2", this, err );
                    this.log.info( "data is %1", data.toString("hex").replace(/(..)/g, "\$1 ") );
                } finally {
                    e.deferNotifies( false );
                }

                // If any zone is on, system entity will indicate power on
                let anyOn = power !== 0;
                if ( ! anyOn ) {
                    for ( let key in this.entities ) {
                        let e = this.entities[ key ];
                        if ( e.hasCapability( 'x_htdlync_zone' ) && e.getAttribute( 'power_switch.state' ) ) {
                            anyOn = true;
                            break;
                        }
                    }
                }
                this.getSystemEntity().setAttributes({
                    'power_switch.state': anyOn,
                    'av_repeat.repeat_mode': ( data[2] & 0x10 | data[3] & 0x10 ) ? "all" : "off",
                    'av_repeat.shuffle': false
                });
            } else {
                this.log.info( "%1 skipping update for zone %2, doesn't exist", this, zoneid );
            }
        } else if ( 0x09 === typ ) {
            // MP3 Play End
            this.log.debug( 5, "%1 handling Play End", this );
            this._mark_mp3_stopped();
        } else if ( 0x0d === typ ) {
            // Zone name - data 0-10 is name; 11 is zone (dup); 12 unused
            let ix = data.indexOf( 0x00 );
            const name = data.slice( 0, ix < 0 || ix > 10 ? 11 : ix );
            this.log.debug( 5, "%1 handling zone name %2=%3", this, zoneid, name );
            const eid = `zone${zoneid}`;
            const e = this.findEntity( eid );
            if ( e ) {
                e.deferNotifies( true );
                if ( ( e.getName() || eid ) === eid ) {
                    e.setName( name.toString() );
                }
                e.setAttribute( 'x_htdlync_zone.name', name.toString() );
                e.deferNotifies( false );
            } else {
                this.log.info( "%1 skipping update for zone %2, doesn't exist", this, zoneid );
            }
        } else if ( 0x0e === typ ) {
            // Zone Source name - data 0-10 is name; 11 is source ID; 12 unused
            const sid = data[11] + 1;
            let ix = data.indexOf( 0x00 );
            const name = data.slice( 0, ix < 0 || ix > 10 ? 11 : ix );
            this.log.debug( 5, "%1 handling zone %2 source name %3=%4", this, zoneid, sid, name );
            const eid = `zone${zoneid}`;
            const e = this.findEntity( eid );
            if ( e ) {
                try {
                    e.deferNotifies( true );
                    e.setAttribute( `x_htdlync_zone.source${sid}`, name.toString() );
                    if ( ! zoneSources[ zoneid ] ) {
                        zoneSources[ zoneid ] = [];
                    }
                    zoneSources[ zoneid ][ sid ] = name.toString();

                    // Name of current input may have changed with this operation, so update it.
                    const curInp = e.getAttribute( 'x_htdlync_zone.input' );
                    e.setAttribute( 'media_source.selected', e.getAttribute( `x_htdlync_zone.source${curInp+1}` ) || "unknown" );
                } finally {
                    e.deferNotifies( false );
                }
            } else {
                this.log.info( "%1 skipping update for zone %2, doesn't exist", this, zoneid );
            }
        } else if ( 0x10 === typ ) {
            // Firmware
            this.log.debug( 5, "%1 handling firmware ident", this );
            const e = this.getSystemEntity();
            e.setAttribute( 'x_htdlync.firmware', data.slice( 0, data.length - 1 ).toString() );
        } else if ( 0x11 === typ ) {
            // MP3 Filename
            this.log.debug( 5, "%1 handling MP3 filename", this );
            let e = this.getSystemEntity();
            e.setAttributes({
                'current_title': data.slice( 0, data.length - 1 ).toString(),
                'state': 'playing'
            }, 'av_transport' );
        } else if ( 0x12 === typ ) {
            // MP3 Artist Name
            this.log.debug( 5, "%1 handling MP3 artist name", this );
            let e = this.getSystemEntity();
            e.setAttribute( 'av_transport.current_artist', data.slice( 0, data.length - 1 ).toString() );
        } else if ( 0x13 === typ ) {
            // MP3 ON
            this.log.debug( 5, "%1 MP3 ON", this );
            let e = this.getSystemEntity();
            e.setAttribute( 'x_htdlync.mp3_ready', true );
        } else if ( 0x14 === typ ) {
            // MP3 OFF
            this.log.debug( 5, "%1 MP3 OFF", this );
            this._mark_mp3_stopped();
            let e = this.getSystemEntity();
            e.setAttribute( 'x_htdlync.mp3_ready', false );
        } else {
            this.log.notice( "%1 no handler for packet type %2 (%3)", this, typ,
                packet.toString("hex").replace(/(..)/g, "\$1 ") );
        }
    }

    _mark_mp3_stopped() {
        let e = this.getSystemEntity();
        e.setAttributes({
            'state': 'stopped',
            'current_title': null,
            'current_artist': null
        }, 'av_transport' );
    }

    action_power_switch_set( entity, params ) {
        if ( entity.hasCapability( 'x_htdlync_zone' ) ) {
            const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
            this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, params.state ? 0x57 : 0x58 ] ) );
        } else {
            this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, params.state ? 0x55 : 0x56 ] ) );
        }
        return Promise.resolve();
    }

    action_power_switch_on( entity, params ) {
        return this.action_power_switch_set( entity, { state: true } );
    }

    action_power_switch_off( entity, params ) {
        return this.action_power_switch_set( entity, { state: false } );
    }

    action_toggle_toggle( entity, params ) {
        return this.action_power_switch_set( entity, { state: ! entity.getAttribute( 'power_switch.state' ) } );
    }

    action_muting_set( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        /* Zone must be on for mute change to work. Zone power off resets mute off. It is what it is. See also set_dnd below. */
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, params.state ? 0x1e : 0x1f ] ) );
        return Promise.resolve();
    }

    action_muting_mute( entity, params ) {
        return this.action_muting_set( entity, { state: true } );
    }

    action_muting_unmute( entity, params ) {
        return this.action_muting_set( entity, { state: false } );
    }

    action_muting_toggle( entity, params ) {
        return this.action_muting_set( entity, { state: ! entity.getAttribute( 'muting.state' ) } );
    }

    action_sys_system_restart( entity, params ) {
        const self = this;
        return new Promise( (resolve) => {
            self.log.notice( "%1 action sys_system.restart: recycling connection", self );
            self._recycle();
            resolve();
        });
    }

    action_media_source_set( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        let s = parseInt( params.input );
        if ( isNaN( s ) ) {
            s = zoneSources[ zoneid ].indexOf( params.input );
            if ( s < 0 ) {
                return Promise.reject( new Error( "Invalid media source" ) );
            }
        }
        const b = s + ( s > 12 ? 0x56 : 0x0f );  // source 1-12 = 0x10-0x1b, source 13-18 = 0x63-0x68
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, b ] ) );
        return Promise.resolve();
    }

    action_volume_set( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        const v = Math.max( 0, Math.min( 1, util.coalesce( parseFloat( params.value ), 0 ) ) );
        // Range is 0 to -61db, 0 is loudest (0xC3), -61 is silent (0x00)
        const b = 0xC3 + Math.floor( v * 61 + 0.5 );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x15, b & 0xff ] ) );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, 0x57 ] ) ) ;  // query, because update isn't automatic
        return Promise.resolve();
    }

    action_volume_set_db( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        const v = Math.max( -61, Math.min( 0, util.coalesce( parseFloat( params.db ), -61 ) ) );
        const b = ( 0xC3 + Math.floor( v + 61.5 ) ) & 0xff;
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x15, b ] ) );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, 0x57 ] ) );  // query, because update isn't automatic
        return Promise.resolve();
    }


    action_media_navigation_next_track( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, 0x0a ] ) );
        return Promise.resolve();
    }

    action_media_navigation_previous_track( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, 0x0c ] ) );
        return Promise.resolve();
    }

    action_av_transport_stop( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, 0x0d ] ) );
        this._mark_mp3_stopped();
        return Promise.resolve();
    }

    action_av_transport_play( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, 0x0b ] ) );
        return Promise.resolve();
    }

    action_av_transport_pause( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x04, 0x0b ] ) );
        return Promise.resolve();
    }

    action_av_repeat_set_repeat( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x00, 0x01, "all" === params.mode ? 0xff : 0x00 ] ) );
        return Promise.resolve();
    }

    action_av_repeat_repeat_all( entity, params ) {
        return this.action_av_repeat_set_repeat( entity, { mode: "all" } );
    }

    action_av_repeat_repeat_off( entity, params ) {
        return this.action_av_repeat_set_repeat( entity, { mode: "off" } );
    }

    async action_x_htdlync_party_mode( entity, params ) {
        const z = Math.min( 12, Math.max( 0, util.coalesce( parseInt( params.zone ), 0 ) ) );
        let s = parseInt( params.source );
        if ( isNaN( s ) ) {
            s = zoneSources[ z || 1 ].indexOf( params.source );
            if ( s < 0 ) {
                return Promise.reject( new Error( "Invalid media source" ) );
            }
        }
        s = Math.min( 18, Math.max( 1, s ) );
        const b = s + ( s > 12 ? 0x5c : 0x35 );  // source 1-12 = 0x36-0x41, source 13-18 = 0x69-0x6e
        this._sendData( Buffer.from( [ 0x02, 0x00, z, 0x04, b ] ) );
    }

    async action_x_htdlync_refresh( entity, params ) {
        this._sendData( Buffer.from( [ 0x02, 0x00, 0x01, 0x0C, 0x00 ] ) );
    }

    async action_x_htdlync_set_all_volume( entity, params ) {
        for ( let eid in this.entities ) {
            const e = this.entities[ eid ];
            if ( e.hasCapability( 'x_htdlync_zone' ) ) {
                this.action_volume_set( e, params );
            }
        }
    }

    async action_x_htdlync_set_all_volume_db( entity, params ) {
        for ( let eid in this.entities ) {
            const e = this.entities[ eid ];
            if ( e.hasCapability( 'x_htdlync_zone' ) ) {
                this.action_volume_set_db( e, params );
            }
        }
    }

    /** Set source name for all zones */
    async action_x_htdlync_set_all_source_name( entity, params ) {
        for ( let eid in this.entities ) {
            const e = this.entities[ eid ];
            if ( e.hasCapability( 'x_htdlync_zone' ) ) {
                await this.action_x_htdlync_zone_set_source_name( e, params );
            }
        }
    }

    async action_x_htdlync_zone_set_source_name( entity, params ) {
        /** Source Name setting is 16 bytes. Usual Head (0x02), reserved (0x00), zone (1 byte), command (0x07),
         *  followed by source address in Data1; Data2-11 are 10 char name with 0x00 terminator (so always 11 bytes),
         *  followed by 0x00. Note that HTD doc says name is 10 bytes in Data2-12, but they can't count. It's 2-11.
         */
        const zone = entity.getAttribute( "x_htdlync_zone.zoneid" );
        let source = parseInt( params.source );
        if ( Number.isNaN( source ) || source < 1 || source > 18 ) {
            throw new RangeError( "Invalid source number" );
        }
        const nom = Buffer.from( ( params.name || String(source) ).substr( 0, 10 ) ); // max 10 bytes

        // Create command Buffer
        const cmd = Buffer.allocUnsafe(16).fill( 0x00 );  // zero-filled buffer, 16 bytes
        Buffer.from( [ 0x02, 0x00, zone, 0x07, source ] ).copy( cmd );  // copy command
        nom.copy( cmd, 5 );  // append new name
        this._sendData( cmd );  // ship it!
        if ( false !== this.config.source_query_offset_hack ) {
            --source;  // WTF. Query requires zero-based source number; everywhere else, it's one-based.
        }
        this._sendData( Buffer.from( [ 0x02, 0x00, zone, 0x0E, source ] ) );  // source name query
    }

    async action_x_htdlync_zone_set_zone_name( entity, params ) {
        /** Zone Name setting is 16 bytes. Usual Head (0x02), reserved (0x00), zone (1 byte), command (0x06),
         *  followed by 0x00 in Data1; Data2-11 are 10 char name with 0x00 terminator (so always 11 bytes),
         *  followed by 0x00. Note that HTD doc says name is 10 bytes in Data2-12, but they can't count. It's 2-11.
         */
        const zone = entity.getAttribute( "x_htdlync_zone.zoneid" );
        const nom = Buffer.from( ( params.name || `Zone ${zone}` ).substr( 0, 10 ) ); // max 10 bytes

        // Create command Buffer
        const cmd = Buffer.allocUnsafe(16).fill( 0x00 );  // zero-filled buffer, 16 bytes
        Buffer.from( [ 0x02, 0x00, zone, 0x06, 0x00 ] ).copy( cmd );  // copy command
        nom.copy( cmd, 5 );  // append new name
        this._sendData( cmd );  // ship it!
        this._sendData( Buffer.from( [ 0x02, 0x00, zone, 0x0D, 0x00 ] ) );  // zone name query
    }

    async action_x_htdlync_zone_set_dnd( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        /* Like mute, zone must be on for DND change to work. Zone power off resets DND to off. Sigh. */
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x04, params.dnd ? 0x59 : 0x5a  ] ) );
    }

    async action_x_htdlync_zone_set_balance( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x16, Math.min( 18, Math.max( -18, parseInt( params.balance || 0 ) ) ) & 0xff ] ) );
        await this._zone_refresh_workaround( entity );
    }

    async action_x_htdlync_zone_set_treble( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        // Note range of transmitted value is different from documentation
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x17, Math.min( 10, Math.max( -10, parseInt( params.treble || 0 ) ) ) & 0xff ] ) );
        await this._zone_refresh_workaround( entity );
    }

    async action_x_htdlync_zone_set_bass( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        // Note range of transmitted value is different from documentation
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x18, Math.min( 10, Math.max( -10, parseInt( params.bass || 0 ) ) ) & 0xff ] ) );
        await this._zone_refresh_workaround( entity );
    }

    // Although docs say this is per-zone, it's actually system-wide/all zones on my firmware.
    async action_x_htdlync_zone_set_names_default( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x1c, 0x00  ] ) );
    }

    async action_x_htdlync_zone_set_audio_default( entity, params ) {
        const zoneid = entity.getAttribute( 'x_htdlync_zone.zoneid' );
        this._sendData( Buffer.from( [ 0x02, 0x00, zoneid, 0x1e, 0x00  ] ) );
        await this._zone_refresh_workaround( entity );
    }

    performOnEntity( entity, actionName, params, ...rest ) {
        // Just log the action and do the system default thing...
        this.log.info( "%1 perform %2 on %3 with %4", this, actionName, entity, params );
        return super.performOnEntity( entity, actionName, params, ...rest );
    }
};
