/** This file is part of Reactor.
 *  Copyright (c) 2021-2025 Kedron Holdings LLC, All Rights Reserved.
 *  Reactor is not public domain or open source. Distribution or derivative works are expressly prohibited.
 */

/**
 * TO-DO
 *
 */

const version = 25304;

const path = require( "path" );

const Logger = require( "server/lib/Logger" );
Logger.getLogger( 'MQTTController', 'Controller' ).always( "Module MQTTController v%1", version );

const Configuration = require("server/lib/Configuration");
const Capabilities = require( "server/lib/Capabilities" );
//const confdir = Configuration.getConfig( "reactor.confdir" );
const dumpdir = Configuration.getConfig( "reactor.logsdir" );

const Controller = require("server/lib/Controller").requires( 22304 );
const Entity = require("server/lib/Entity").requires( 22306 );

const TimerBroker = require("server/lib/TimerBroker");

const Mutex = require( "server/lib/Mutex" );

const AlertManager = require( "server/lib/AlertManager" );

const util = require("server/lib/util");
const coalesce = util.coalesce;
const isUndef = util.isUndef;

const lexp = require('common/lexp');

// Ref: https://github.com/mqttjs/MQTT.js#readme
const mqtt = require( 'mqtt' );
const MQTT_REQUIRED = 51401;  // 5.14.1

const { hostname, EOL } = require( 'os' );

const DEFAULT_CONNECTTIMEOUT = 10000;
const DEFAULT_RECONNECTPERIOD = 5000;
const MAX_FAIL = 3;

const VALID_ID = /^[\p{Alphabetic}_$-][\p{Alphabetic}0-9_$-]*$/iu;

var impl = false;
var expr_ce = {};
var localCapabilities = new Map();

var parseBoolean = function( s ) {
    if ( "null" === s || null === s ) {
        return null;
    } else if ( "boolean" === typeof s ) {
        return s;
    } else if ( "string" === typeof s ) {
        return null !== s.match( /^(1|true|yes|on)$/i );
    } else if ( "number" === typeof s ) {
        return s !== 0;
    }
    return !!s;
};

module.exports = class MQTTController extends Controller {

    static mtx_once = new Mutex();

    constructor( struct, id, config ) {
        super( struct, id, config );

        this.stopping = false;
        this.my_ident = ( this.config.ident || id ).replace( /\s+/g, "" ).replace( /[^a-z0-9_-]/ig, "_" );
        this.topic_prefix = `reactor/${this.my_ident}/`;
        this.client = false;
        this.connectTimer = false;
        this.retries = 0;

        this.events = {};
        this.ev_patterns = new Map();
        this.inits = [];
        this.ext_patterns = [];
        this.outgoing = {};

        this.poll_queue = [];
        this.last_poll = 0;
        this.pollTimer = false;

        this.config.entities = this.config.entities || {};
        this.config.echo = coalesce( this.config.echo, false );
        this.config.publish_entities = coalesce( this.config.publish_entities, this.config.echo );
        this.config.publish_rule_states = coalesce( this.config.publish_rule_states, this.config.echo );
        this.config.publish_global_variables = coalesce( this.config.publish_global_variables, this.config.echo );

        this.default_qos = coalesce( parseInt( this.config.default_qos ), 0 );
        this.default_retain = parseBoolean( this.config.default_retain );

        this.echo_qos = coalesce( parseInt( this.config.echo?.qos ), this.default_qos );
        this.echo_retain = ( "boolean" === typeof this.config.echo?.retain ) ? this.config.echo.retain :
            this.default_retain;

        this.poll_frequency_ms = Math.max( 100, coalesce( parseInt( this.config.poll_frequency ), 5000 ) );
    }

    /** Start the controller.
     */
    async start() {
        if ( "undefined" === typeof mqtt.MQTTJS_VERSION || String( mqtt.MQTTJS_VERSION ).startsWith( "4." ) ) {
            throw new Error(
                "Invalid mqtt package version; please update packages by running `npm run deps` in your MQTTController install directory."
            );
        }

        try {
            let v = ( mqtt.MqttClient?.VERSION || "0.0.0" ).split( '.' );
            v = coalesce( parseInt( v[0] ), 0 )*10000 +
                coalesce( parseInt( v[1] ), 0 )*100 +
                coalesce( parseInt( v[2] ), 0 );
            this.log.debug( 5*0, "%1 mqtt package version is %2 (%3)", this, mqtt.MqttClient?.VERSION, v );
            if ( v < MQTT_REQUIRED ) {
                AlertManager.addAlert( AlertManager.WARN,
                    `The version of the "mqtt" third-party support package currently installed for use by MQTTController is out of date. Please see the MQTTController installation instructions for upgrade information.`,
                    { tag: 'mqttcontroller_mqtt_version' }
                );
            } else {
                AlertManager.dismissTag( 'mqttcontroller_mqtt_version' );
            }
        } catch ( err ) {
            this.log.exception( err );
        }

        /* Load implementation data if not yet loaded; multiple instances are common, so use a mutex to ensure
         * that only one instance loads the data (i.e. another doesn't start trying when an async block of the first
         * is waiting.)
        */
        try {
            await MQTTController.mtx_once.acquire();
            if ( !impl ) {
                await this._load_implementation();
            }
        } catch ( err ) {
            this.log.exception( err );
        } finally {
            MQTTController.mtx_once.release();
        }

        this.log.info( "%1 instance topic ident is %2", this, this.my_ident );

        let sys = this.getSystemEntity();
        sys.extendCapability( 'x_mqtt' );
        if ( sys._impl_hash !== impl.hash ) {
            sys.refreshCapability( 'x_mqtt' );
            sys._impl_hash = impl.hash;
        }
        sys.setAttributes({
            version: version,
        }, 'x_mqtt' );

        /* Re-initialize instance */
        this.events = {};
        this.ev_patterns.clear();
        this.inits = [];
        this.poll_queue = [];

        const polled = new Set();

        /* We do not remove external subscriptions. A restart of the controller should resume with
         * the subscriptions in place.
         */
        // this.ext_patterns = [];
        this.stopping = false;

        /* Initialize enumerated entities */
        let allEntities = {};
        Object.keys( this.entities ).forEach( id => allEntities[ id ] = true );
        delete allEntities.system;
        delete allEntities.controller_all;
        for ( let eid of Object.keys( this.config.entities || {} ) ) {
            if ( ! Entity.isValidID( eid ) ) {
                this.log.err( "%1 configuration error: invalid entity ID '%2'", this, eid );
                this.sendError( "Configuration error: invalid entity ID {0:q}", eid );
                continue;
            }
            this.log.debug( 5, "%1 configuring entity %2", this, eid );
            let ec = this.config.entities[ eid ];
            let tpl = {};
            if ( ! ec.include ) {
                ec.include = [];
            } else if ( "string" === typeof ec.include ) {
                ec.include = [ ec.include ];
            } else if ( ! Array.isArray( ec.include ) ) {
                this.log.err( "%1 entity %2 include must be a string or array of strings", this, eid );
                this.sendError( "Entity {0:q} invalid 'include' in configuration", eid );
                continue;
            }
            let seen = new Set();
            for ( let nom of ( ec.include || [] ) ) {
                if ( "string" === typeof nom ) {
                    try {
                        seen.add( nom );
                        tpl = this._include_template( tpl, nom, seen, [`Entity ${eid} configuration`] );
                    } catch ( err ) {
                        this.log.err( "%1 entity %2 failed to include template %3: %4", this, eid, nom, err );
                        this.sendError( "Configuration for {0:q} failed to include template {1:q}: {2}",
                            eid, nom, String( err ) );
                    }
                }
            }
            //seen.clear(); // don't do this -- we apply it to an attribute later for visibility/troubleshooting.

            // Check that required fields for template have been included in entity definition.
            if ( Array.isArray( tpl.requires ) ) {
                let missing = [];
                for ( let nom of tpl.requires ) {
                    if ( isUndef( ec[ nom ] ) ) {
                        missing.push( nom );
                    }
                }
                if ( 0 !== missing.length ) {
                    this.log.err( "%1 entity %2 config missing %3 required by template", this, eid, missing.join(', ') );
                    this.sendError( "Configuration for {0:q} is missing the following required template field(s): {1}",
                        eid, missing.join(', ') );
                    continue;
                }
            }

            // Merge configuration into template.
            tpl = this._merge( tpl, { ...ec } );  /* entity configuration may override template values */
            this.log.debug( 5, "%1 merged configuration for %2:", this, eid );
            if ( this.log.getEffectiveLevel() >= 5 ) {
                this.log.raw( JSON.stringify( tpl, null, 4 ) );
            }

            let e = this.findEntity( eid );
            if ( ! e ) {
                this.log.debug( 5, "%1 created new entity for %2", this, eid );
                e = this.getEntity( tpl.type || 'Entity', eid );
            } else {
                e.deferNotifies( true );
            }

            try {
                e.__template = tpl;
                e.markDead( false );
                delete allEntities[ eid ];

                e.setName( ec.name || eid );  /* specifically not tpl.name here */
                e.setType( tpl.type || 'Entity' );

                let cx = [];
                for ( let cap of ( tpl.capabilities || [] ) ) {
                    if ( localCapabilities.has( cap ) ) {
                        try {
                            /* Attach copy to entity. Send in the clones. */
                            e.extendCapability( cap,
                                util.clone( localCapabilities.get( cap ).capability ), true ); // extend definition
                            cx.push( cap );  // only if extend succeeds
                        } catch ( err ) {
                            this.log.err( "%1 unable to extend %2 for %3 due to an error: %4", this, cap, eid, err );
                            this.log.exception( err );
                            this.sendError( "Configuration for {0:q} failed because capability {1:q} could not be extended due to an error. Please see the log.",
                                eid, cap );
                        }
                    } else {
                        cx.push( cap );
                    }
                }
                cx.unshift( 'x_mqtt_device' );
                // N.B.: below removes capabilities not in cx
                const [ newCaps, delCaps, errCaps ] = e.extendCapabilities( cx );  // eslint-disable-line no-unused-vars
                if ( errCaps?.length ) {
                    this.log.warn( "%1 entity %2 unable to configure capabilities %3", this, e, errCaps );
                    this.sendWarning( "Configuration for {0:q} is incomplete because the following requested capabilities are undefined: {1}",
                        eid, errCaps.join( ', ' ) );
                }
                if ( e._impl_hash !== impl.hash || e._tpl_version !== tpl.version ) {
                    this.log.notice( "%1 refreshing capabilities for %2; capability change detected", this, e );
                    e.refreshCapabilities();
                    e._impl_hash = impl.hash;
                    e._tpl_version = tpl.version;
                }
                e.setAttribute( 'x_mqtt_device.templates', Array.from( seen.keys() ) );

                // Set primary attribute.
                if ( "string" === typeof tpl.primary_attribute ) {
                    try {
                        if ( ! e.hasAttribute( tpl.primary_attribute ) ) {
                            /* Not yet present; try to define it. */
                            e.setAttribute( tpl.primary_attribute, null );
                        }
                        e.setPrimaryAttribute( tpl.primary_attribute );
                        this.log.debug( 5, "%1 set primary attribute of %2 to %3", this, tpl.primary_attribute, e );
                    } catch ( err ) {
                        this.log.warn( "%1 can't set primary attribute %2 on %3: %4", this, tpl.primary_attribute, e, err );
                        this.sendWarning( "Can't set primary attribute to {0:q} on {1:q}: {2}", tpl.primary_attribute,
                            eid, String( err ) );
                    }
                }

                /* Subscribe to other events per the template */
                e.__events = {};
                if ( ! isUndef( tpl.events ) ) {
                    if ( null !== tpl.events && "object" !== typeof tpl.events ) {
                        this.log.err( '%1 template for %2 has improperly structured "events" section', this, eid );
                        this.sendError( "Configuration for {0:q} has improperly structured {1:q} section.", eid, "events" );
                    } else {
                        for ( let topic of Object.keys( tpl.events || {} ) ) {
                            const ev = topic.trim().replace( /%([^%]+)%/g, ( m, p ) => {
                                let altstr = "";
                                const k = p.indexOf( ':' );
                                if ( k >= 0 ) {
                                    altstr = p.substring( k + 1 );
                                    p = p.substring( 0, k );
                                }
                                return String( coalesce( ec[p], altstr ) );
                            });
                            this.log.debug( 5, "%1 registering event handler %3 for %2", this, eid, topic );
                            //let attr = util.clone( tpl.events[ topic ] );  // deep copy, we may make changes.
                            let attr = util.clone( tpl.events[ topic ] );  // deep copy, we may make changes.
                            if ( "object" === typeof attr && ! Array.isArray( attr ) ) {
                                /* Convert key/value map form to array form for live use */
                                attr = [];
                                for ( let [k,v] of ( Object.entries( tpl.events[ topic ] ) || {} ) ) {
                                    if ( null === v || "object" !== typeof v ) {
                                        attr.push( { attribute: k, value: v } );
                                    } else {
                                        v.attribute = k;
                                        attr.push( v );
                                    }
                                }
                                // Key/value map form can specify "sort" numeric value to control order of
                                // evaluation; unspecified/NaN sorts last.
                                attr.sort( (a, b) => coalesce(parseInt(a.sort), 32767) - coalesce(parseInt(b.sort), 32767) );
                                this.log.debug( 7, "%1 converted key/value map form for %2 from %3 to %4", this, topic,
                                    tpl.events[ topic ], attr );
                            }
                            if ( ev.endsWith( '/#' ) || ev.match( /\/\+\// ) ) {
                                /* Wildcard topic (slow) */
                                if ( ! this.ev_patterns.has( ev ) ) {
                                    let pattern = '^' + ev.replace( /[.?*]/g, "\\$&" ) + '$';
                                    pattern = pattern.replace( /\/\+/g, "/[^/]+" );
                                    pattern = pattern.replace( /\/#/g, "/.*" );
                                    let re = new RegExp( pattern );
                                    this.ev_patterns.set( ev, { pattern: pattern, regexp: re, entities: [ eid ] } );
                                } else {
                                    this.ev_patterns.get( ev ).entities.push( eid );
                                }
                            } else {
                                /* Fast-map topics, no wildcards */
                                this.events[ev] = this.events[ev] || [];
                                this.events[ev].push( eid );
                            }
                            e.__events[ ev ] = attr;
                        }
                    }
                }

                /* Subscribe to LWT for this topic, directing to this entity. */
                if ( "string" === typeof tpl?.lwt ) {
                    const lwt = tpl.lwt = tpl.lwt.replace( /%([^%]+)%/g, ( m, p ) => {
                        let altstr = "";
                        const k = p.indexOf( ':' );
                        if ( k >= 0 ) {
                            altstr = p.substring( k + 1 );
                            p = p.substring( 0, k );
                        }
                        return String( coalesce( ec[p], altstr ) );
                    });
                    e.__events[ lwt ] = this.handle_lwt.bind( this );
                    this.events[ lwt ] = this.events[ lwt ] || [];
                    this.events[ lwt ].push( eid );
                }

                /* Action implementations from the template. Start by building a Set of all actions from all
                 * extended capabilities. As implementations are registered, remove from the set. Whatever
                 * is left has no implementation and we'll mark it unavailable.
                 */
                e.__actions = {};
                const allActions = new Set();
                for ( let cap of e.getCapabilities() ) {
                    const cdef = e.getCapability( cap );
                    Object.keys( cdef.actions || {} ).forEach( act => allActions.add( `${cap}.${act}` ) );
                }
                if ( ! isUndef( tpl.actions ) ) {
                    if ( null !== tpl.actions && "object" !== typeof tpl.actions ) {
                        this.log.err( '%1 template for %2 has improperly structured "actions" section', this, eid );
                        this.sendError( "Configuration for {0:q} has improperly formed {1:q} section.", eid, "actions" );
                    } else {
                        for ( let cap of Object.keys( tpl.actions || {} ) ) {
                            const cdef = e.getCapability( cap );
                            if ( ! cdef ) {
                                this.log.err( "%1 entity %2 actions configuration specifies unrecognized capability %3",
                                    this, e, cap );
                                this.sendError( "Configuration for {0:q} has unrecognized capability {1:q} in actions",
                                    eid, cap );
                                continue;
                            }
                            for ( let act of Object.keys( tpl.actions[cap] || {} ) ) {
                                const key = cap + '.' + act;
                                const adef = tpl.actions[cap][act];
                                this.log.debug( 5, "%1 registering action %2 for %3 as %4", this, key, eid, adef );
                                try {
                                    if ( "object" === typeof adef && cap.startsWith( 'x_' ) ) {
                                        // Extended capability can define extended action with arguments
                                        if ( ! cdef.actions ) {
                                            cdef.actions = {};
                                        }
                                        if ( ! cdef.actions[act] ) {
                                            cdef.actions[act] = { arguments: adef.arguments || {} };
                                        }
                                    }
                                    e.__actions[ key ] = adef;
                                    e.registerAction( key, ( "boolean" === typeof adef ) ? adef : true );
                                    allActions.delete( key );
                                } catch ( err ) {
                                    this.log.err( "%1 failed to register action %2 for %3: %4", this, key, eid, err );
                                }
                            }
                        }
                    }
                }
                if ( allActions.size > 0 ) {
                    for ( let act of allActions ) {
                        if ( "function" !== typeof this[ `action_${act.replace( /\./g, "_" )}` ] ) {
                            e.registerAction( act, false );
                            this.log.debug( 5, "%1 action %3 for %2 not defined, declared unavailable", this, eid, act );
                        } else {
                            this.log.debug( 5, "%1 action %3 for %2 provided by default function", this, eid, act );
                        }
                    }
                }
                allActions.clear(); // free space

                if ( tpl.query ) {
                    this.inits.push( e );
                    if ( tpl.poll_interval ) {
                        polled.add( eid );
                    }
                }
            } catch ( err ) {
                this.log.err( "%1 error initializing entity %2: %3", this, eid, err );
                this.log.exception( err );
                this.sendError( "An error occurred while initializing {0:q}; please see the log for details.", eid );
            } finally {
                e.deferNotifies( false );
            }
        }

        Object.keys( allEntities ).forEach( id => {
            let e = this.entities[ id ];
            if ( e ) {
                e.markDead();
                this.log.notice( "%1 device %2 no longer available, marking %3 for removal", this, id, e );
                this.sendWarning( "Controller {0} device {1:q} ({2}) no longer exists.", this.getID(), id, e.getName() );
            }
        });
        this.purgeDeadEntities();

        try {
            this.subscribe( AlertManager );
        } catch ( err ) {
            this.log.err( '%1 unable to subscribe to alerts: %2', this, err );
        }

        return new Promise( ( resolve ) => {
            this.run();

            if ( this.config.echo ) {
                this.getStructure().waitAllControllers( 60000, this ).then( data => {
                    if ( false !== this.config.publish_entities ) {  /* default publish */
                        this.subscribe_entities();
                    }
                    if ( false !== this.config.publish_global_variables ) {
                        this.subscribe_global_vars();
                    }
                    if ( false !== this.config.publish_rule_states ) {
                        this.subscribe_rules();
                    }
                    return data;
                });
            }

            if ( polled.size > 0 ) {
                this._start_polling( polled );
            }

            resolve( this );
        });
    }

    async stop() {
        this.stopping = true;
        this.log.notice( "%1 stopping", this );

        this._clearConnectTimer();

        if ( this.pollTimer ) {
            this.pollTimer.cancel();
        }
        this.poll_queue = [];

        if ( this.client ) {
            for ( let mid of Object.keys( this.outgoing ) ) {
                try {
                    clearTimeout( this.outgoing[ mid ] );
                    this.client.removeOutgoingMessage( mid );
                } catch ( err ) {  // eslint-disable-line no-unused-vars
                    // Nada
                }
            }
            try {
                this.log.info( "%1 sending LWT", this );
                await this._publish( `${this.topic_prefix}LWT`, 'offline', {
                    qos: coalesce( parseInt( this.config.lwt_qos ), 1 ),
                    retain: coalesce( this.config.lwt_retain, true )
                });
            } catch ( err ) {  // eslint-disable-line no-unused-vars
                /* Nada */
            } finally {
                this.log.info( "%1 closing broker connection", this );
                this.client.end( true );
                this.client = false;
            }
        }

        this.outgoing = {};
        this.events = {};
        this.inits = [];
        /* We do not remove external subscriptions. A restart of the controller should resume with
         * the subscriptions in place.
         */
        // this.ext_patterns = [];

        return await super.stop();
    }

    run() {
        if ( this.stopping ) {
            return;
        }
        if ( ! this.client ) {
            const self = this;
            const url = self.config.source || "mqtt://127.0.0.1:1883";
            self.log.notice("%1 connecting to broker at %2", self, url);
            let copt = self.config.options || {};
            if ( self.config.username ) {
                copt.username = self.config.username;
                copt.password = self.config.password || "";
            }
            copt.clientId = copt.clientId || `reactor_${process.env.HOST || hostname()}_${this.my_ident}`;
            copt.reconnectPeriod = coalesce( parseInt( copt.reconnectPeriod ), DEFAULT_RECONNECTPERIOD );
            copt.connectTimeout = coalesce( copt.connectTimeout, DEFAULT_CONNECTTIMEOUT );
            copt.resubscribe = false;
            /** This is the default LWT delay. If we don't reconnect in this period, the broker should automatically
             *  send the LWT for us. This is one full cycle of reconnect delay + connection timeout, plus padding.
             */
            const wdi = Math.floor( copt.reconnectPeriod + copt.connectTimeout + 1000 ) / 1000;
            copt.will = {
                topic: `${this.topic_prefix}LWT`,
                payload: 'offline',
                qos: coalesce( this.config.lwt_qos, 1 ),
                retain: coalesce( this.config.lwt_retain, true ),
                properties: {
                    willDelayInterval: coalesce( parseInt( this.config.lwt_delay ), wdi )
                }
            };
            // copt.rejectUnauthorized = ?;  // for ref/doc only, use with mqtts://, user can provide in "options"

            self.log.debug( 5, "%1 connecting to %2 with options %3", self, url, copt );
            /** Not sure if it's openLuup's MQTT broker (likely) or the MQTT client library, but on reconnect, the
             *  session appears to never get completely set up and "connect" never comes. So, set up a timer to
             *  babysit it. Seems like both need a fix in this instance; mqtt should not stall, regardless of what the
             *  broker may do to it.
             */
            self._startConnectTimer( 2 * copt.connectTimeout );
            self.client = mqtt.connect( url, copt );
            self.client.on( "connect", ( rc ) => {
                self._clearConnectTimer();
                self.log.info( "%1 connected; opening subscriptions", self );
                self.log.debug( 7, "%1 connect callback(%2)", self, rc );
                self.client.on( "message", ( topic, payload, packet ) => {
                    if ( self.log.getEffectiveLevel() >= 6 ) {
                        let p = {};
                        for ( let kk in packet ) {
                            p[kk] = packet[kk];
                        }
                        self.log.debug( 6, "%1 message packet=%2", self, p );
                    }
                    if ( self.config.log_topics ) {
                        /* Match all topics if (boolean) true; otherwise must be array to match (elements are topic strings) */
                        const match = ( true === self.config.log_topics ) ||
                            ( Array.isArray( self.config.log_topics ) && self.config.log_topics.find( el => topic.match( el ) ) );
                        if ( match ) {
                            try {
                                const { writeFileSync } = require( 'fs' );
                                const { join } = require( 'path' );
                                writeFileSync( join( dumpdir, `${this.getID()}-topics.log` ),
                                    `${new Date().toISOString()} "${topic}" ${packet.qos} ${packet.retain} ${payload}${EOL}`, { flag: "as", encoding: "utf-8" } );
                            } catch ( err ) {
                                self.log.err( "%1 failed to log message for log_topics %2: %3", self, topic, err );
                            }
                        }
                    }
                    /* Dispatch */
                    if ( topic.startsWith( this.topic_prefix ) ) {  /* includes trailing / */
                        /* A control message, or something we published making a round trip. */
                        self.handle_controller_event( topic, payload, packet );
                        return;
                    }

                    /*
                    // ??? This should be configurable.
                    if ( topic.includes( '/discovery/' ) || topic.endsWith( '/announce/' ) ) {  // for tasmota, shelly
                        try {
                            const { writeFileSync } = require( 'fs' );
                            const { join } = require( 'path' );
                            writeFileSync( join( dumpdir, "mqtt-discovery.log" ),
                                payload, { encoding: "utf-8" } );
                        } catch ( err ) {
                            self.log.err( "%1 failed to dump discovery data for %2: %3", self, topic, err );
                        }
                    }
                    */

                    if ( self.events[ topic ] ) {
                        self.log.debug( 5, "%1 topic %2 associated with %3", self, topic, self.events[ topic ] );
                        for ( let eid of self.events[ topic ] ) {
                            self.log.debug( 5, "%1 dispatching %2 to %3", self, topic, eid );
                            try {
                                self.handle_event( eid, topic, topic, payload.toString( 'UTF-8' ), packet );
                            } catch ( err ) {
                                self.log.err( "%1 event handler failed for %2 on %3: %4", self, topic, eid, err );
                                self.log.exception( err );
                            }
                        }
                    }
                    for ( let [ev,p] of self.ev_patterns.entries() ) {
                        self.log.debug( 6, "%1 checking event %2 pattern %3 against topic %4", this, ev, p.pattern, topic );
                        if ( p.regexp.test( topic ) ) {
                            for ( const eid of p.entities ) {
                                self.log.debug( 5, "%1 dispatching %2 (from wildcard event %3) to %4", self, topic, ev, eid );
                                try {
                                    self.handle_event( eid, ev, topic, payload.toString( 'UTF-8' ), packet );
                                } catch ( err ) {
                                    self.log.err( "%1 event handler failed for %2 on %3: %4", self, topic, eid, err );
                                    self.log.exception( err );
                                }
                            }
                        }
                    }
                    if ( self.ext_patterns.length > 0 ) {
                        try {
                            self._handle_ext_event( topic, payload.toString( 'UTF-8' ) );
                        } catch ( err ) {
                            self.log.err( "%1 error while handling external event notifications: %2", self, err );
                            self.log.exception( err );
                        }
                    }
                });
                self.client.subscribe( "#", {
                    qos: coalesce( parseInt( this.config.subscription_qos ), 0 )
                }, ( err, granted ) => {
                    self.log.debug( 5, "%1 subscribe all returns err=%2, granted=%3", self, err, granted );
                    if ( err ) {
                        self.offline();
                        self._recycle();
                        self.log.err( "%1 unexpected broker response to subscription attempt (all): %2", self, err );
                        return;
                    }
                    self.log.info( "%1 sending inits and going online!", self );
                    self.online();
                    self._publish( `${this.topic_prefix}LWT`, "online", { qos: 1, retain: true } ).catch( err => {} );  // eslint-disable-line no-unused-vars
                    self.do_inits();
                });
                try {
                    self.publish_alerts();
                } catch ( err ) {
                    self.log.exception( err );
                }
                self.retries = 0;
            });
            self.client.on( "reconnect", () => {
                self.log.notice( "%1 attempting to reconnect", self );
                if ( 0 === ( ++self.retries % MAX_FAIL ) ) {
                    self.offline();
                    self.log.notice( "%1 too many retries; discarding client", self );
                    self._recycle();
                    return;
                }
                self._startConnectTimer( 2 * copt.connectTimeout );
            });
            self.client.on( "error", ( err ) => {
                self._clearConnectTimer();
                self.log.err( "%1 %2", self, err );
            });
            self.client.on( "offline", () => {
                self._clearConnectTimer();
                self.log.notice( "%1 broker offline!", self );
                if ( self.pollTimer ) {
                    self.pollTimer.cancel();
                }
                if ( false === self.config.reuse_client ) {
                    self.log.notice( "%1 discarding client", self );
                    this._recycle();
                } else if ( 0 === ( ++self.retries % MAX_FAIL ) ) {
                    self.offline();
                    self.log.notice( "%1 too many retries; discarding client", self );
                    self._recycle();
                } else {
                    self.log.notice( "%1 waiting for auto-reconnect", self );
                }
            });
        }
    }

    _include_template( t, nom, seen, parray ) {
        if ( ! impl.templates[ nom ] ) {
            throw new ReferenceError( `Template "${parray[parray.length-1]}" references undefined template "${nom}"` );
        }
        const barray = parray.concat( [ nom ] );  // creates new array
        const pp = barray.join( ' -> ' );
        let tpl = util.clone( impl.templates[ nom ] );
        if ( tpl.requires_version && version < tpl.requires_version ) {
            this.log.warn( "%1: template %2 requires MQTTController %3; this is %4; template may malfunction.",
                this, nom, tpl.requires_version, version );
            this.log.info( "%1: include path is: %2", this, pp );
            this.sendWarning( "The template {0:q} requires MQTTController version {1}, but the running version is only {2}. The template may not function as expected or cause other errors.",
                nom, tpl.requires_version, version );
        }
        if ( "string" === typeof tpl.requires ) {
            tpl.requires = [ tpl.requires ];
        }
        t = this._merge( t, tpl );

        // Handle includes
        let inc = tpl.include;
        if ( "string" === typeof inc ) {
            inc = [ inc ];
        }
        if ( Array.isArray( inc ) ) {
            for ( let inom of inc ) {
                if ( "string" !== typeof inom || ! impl.templates[ inom ] || "object" !== typeof impl.templates[ inom ] ) {
                    this.log.err( "%1 template %2 attempted to include undefined template %3", this, nom, inom );
                    this.log.info( "%1: include path is: %2", this, pp );
                    this.sendError( "Template {0:q} attempts to include undefined template {1:q}", nom, inom );
                } else if ( seen.has( inom ) ) {
                    this.log.debug( 5, "%1 template %2 includes %3, which has already been included (ignored)", this, nom, inom );
                } else {
                    seen.add( inom );
                    t = this._include_template( t, inom, seen, barray );
                }
            }
        } else if ( inc ) {
            this.log.err( "%1 template %2 invalid 'include' data", this, nom );
            this.sendError( "Template {0:q} has invalid 'include' data", nom );
        }

        delete t.poll_interval;  /* Poll interval cannot be in a template */
        return t;
    }

    /* Force close and recycle connection */
    _recycle() {
        try {
            if ( this.client ) {
                this.client.end( true );
            }
        } catch ( err ) {
            this.log.exception( err );
        } finally {
            this._clearConnectTimer();
            this.client = false;
            if ( ! this.stopping ) {
                this.startDelay( coalesce( parseInt( this.config?.options?.reconnectPeriod ),
                    DEFAULT_RECONNECTPERIOD ) );
            }
        }
    }

    _startConnectTimer( timeout ) {
        if ( this.connectTimer ) {
            this._clearConnectTimer();
        }
        this.connectTimer = setTimeout( () => {
            this.log.err( "%1 failed to establish connection", this );
            this.connectTimer = false;
            this._recycle();
        }, timeout || 30000 );
    }

    _clearConnectTimer() {
        if ( this.connectTimer ) {
            try {
                clearTimeout( this.connectTimer );
            } catch ( err ) {
                this.log.debug( 5, "%1 clearing connect timer: %2", this, err );
            }
        }
        this.connectTimer = false;
    }

    _publish( topic, message, options, callback ) {
        /** Note on mqtt package: when publish with qos=0, the packet is not queued outgoing and the callback is called
         *  immediately. This means mid doesn't get set before the callback gets called, so we need to detect undefined
         *  mid in the callback and just know that it's OK. For qos>0, the packet is put into storage and sent, and the
         *  package waits for an ACK from the broker before calling the callback.
         */
        const self = this;
        let mid;
        return new Promise( ( resolve, reject ) => {
            self.client.publish( topic, message, options, ( err, packet ) => {
                self.log.debug( 6, "%1 publish messageId %2 topic %3 result %4", self, mid, topic, err || "success" );
                if ( ! mid ) {
                    resolve();
                    return;
                }
                if ( self.outgoing[ mid ] ) {
                    try {
                        clearTimeout( self.outgoing[ mid ] );
                        delete self.outgoing[ mid ];
                    } catch ( err ) {
                        self.log.exception( err );
                    }
                }
                if ( err ) {
                    if ( "Message removed" !== err.message ) {
                        self.log.err( "%1 send messageId %2 topic %3 failed: %4", self, mid, topic, err );
                    } else {
                        err.cause = "Timeout waiting for ACK";
                    }
                    if ( callback ) {
                        try {
                            callback( err, packet );
                        } catch ( e ) {
                            self.log.err( "%1 error in publish callback for %2 topic %3: %4", self, mid, topic, e );
                            self.log.exception( e );
                        }
                    }
                    reject( err );
                    return;
                }
                if ( callback ) {
                    try {
                        callback( err, packet );
                    } catch ( e ) {
                        self.log.err( "%1 error in publish callback for messageId %2 topic %3: %4", self, mid, topic, e );
                        self.log.exception( e );
                    }
                }
                resolve();
            });

            /** Timeouts only for QOS != 0. For QOS 0, mqtt package calls the callback immediately (before we get here,
             *  in fact).
             */
            if ( 0 !== ( options?.qos || 0 ) ) {
                mid = self.client.getLastMessageId();
                self.log.debug( 6, "%1 tracking published messageId %2 topic %3 qos %4", self, mid, topic, options.qos );
                self.outgoing[ mid ] = setTimeout( () => {
                    delete self.outgoing[ mid ];
                    self.log.err( "%1 messageId %2 topic %3 timeout waiting for broker ACK; aborting publish", self, mid, topic );
                    try {
                        self.client.removeOutgoingMessage( mid );
                    } catch ( err ) {
                        self.log.err( "%1 error removing timed-out messageId %2 topic %3: %4", self, mid, topic, err );
                        self.log.exception( err );
                    }
                }, coalesce( parseInt( self.config.publish_timeout ), 60000 ) );
            }
        });
    }

    /** Merge, for entity configuration. Takes two objects. The contents of obj2 are merged into obj1,
     *  with any overlapping keys being superceded by obj2. Honors subkeys and arrays (that is, existing
     *  values in obj1 in sub-objects and -arrays are merged with those in obj2 through the full depth of
     *  the tree.
     */
    _merge( obj1, obj2 ) {
        function mergeObjects( o1, o2 ) {
            for ( let key of Object.keys( o2 ) ) {
                if ( key.startsWith( '__' ) || "include" === key ) {
                    continue;
                }
                let v1 = o1[key];
                let v2 = o2[key];
                if ( "undefined" === typeof v1 ) {
                    o1[key] = v2;
                } else if ( Array.isArray( v2 ) ) {
                    if ( Array.isArray( v1 ) ) {
                        o1[key] = v1.concat( v2.filter( el => ! v1.includes( el ) ) );
                    } else {
                        throw TypeError( `Incompatible types at ${key}` );
                    }
                } else if ( null !== v2 && "object" === typeof v2 ) {
                    if ( null === v1 ) {
                        o1[key] = v2;
                    } else if ( "object" === typeof v1 ) {
                        o1[key] = mergeObjects( o1[key], v2 );
                    } else {
                        throw TypeError( `Incompatible types at ${key}` );
                    }
                } else {
                    o1[key] = v2;
                }
            }
            return o1;
        }
        return mergeObjects( obj1 || {}, obj2 || {} );
    }

    save_template_data( d, pn, en ) {
        const fpn = path.join( pn, en );
        if ( ! isUndef( d.capabilities ) ) {
            if ( "object" === typeof d.capabilities ) {
                for ( let cap in ( d.capabilities || {} ) ) {
                    if ( ! cap.startsWith( 'x_' ) ) {
                        this.log.err(
                            "%1 file %2 attempts to define capability %3; extension capabilities must begin with 'x_'",
                            this, fpn, cap
                        );
                        this.sendError(
                            `Configuration file {0:q} cannot define {1:q}; the names of extension capabilities must begin with "x_"`,
                            fpn, cap
                        );
                        continue;
                    }
                    const def = d.capabilities[ cap ];
                    if ( ! ( def.attributes || def.actions ) ) {
                        this.log.err(
                            "%1 file %2 attempts to define capability %3 but contains neither attributes nor actions",
                            this, fpn, cap
                        );
                        this.sendError( "Configuration file {0:q} capability {1:q} contains neither attributes nor actions",
                            fpn, cap
                        );
                        continue;
                    }
                    if ( localCapabilities.has( cap ) ) {
                        if ( ! util.deepCompare( def, localCapabilities.get( cap ).capability ) ) {
                            const entry = localCapabilities.get( cap );
                            this.log.err(
                                "%1 file %2 attempts to redefine capability %3 previously defined by %4; ignored",
                                this, fpn, cap, entry.source
                            );
                            this.sendError(
                                "Template file {0:q} attempts to redefine capability {1:q}, previously defined by {2:q}; ignored.",
                                fpn, cap, entry.source
                            );
                        }
                        continue;
                    }
                    this.log.debug( 5, "%1 file %2 defines local capability %3", this, fpn, cap );
                    localCapabilities.set( cap, { source: fpn, capability: def } );
                }
            }
        }
        for ( let te of Object.keys( d.templates || {} ) ) {
            /* Check template ID */
            if ( ! te.match( VALID_ID ) ) {
                this.log.err( "%1 file %2 attempts to define template with invalid ID %3", this, fpn, te );
                continue;
            }
            const tx = d.templates[ te ];
            if ( ! ( tx && "object" === typeof tx && tx.capabilities ) ) {
                this.log.err( "%1 file %2 in %3 template %4 definition invalid", this, en, fpn, te );
                continue;
            }
            if ( impl.templates[ te ] ) {
                const sauce = impl.templates[ te ].__source || "system data";
                this.log.warn( "%1 file %2 overrides existing template definition %3 from %4", this, fpn, te, sauce );
            }
            this.log.debug( 6, "%1 file %2 defines template %3", this, fpn, te );
            tx.__source = fpn;
            impl.templates[ te ] = tx;
        }
    }

    async _load_template_dir( td ) {
        const fs = require( "fs" );

        this.log.debug( 5, "%1 loading template in %2", this, td );
        if ( ! fs.existsSync( td ) ) {
            return;
        }

        const yaml = require( "js-yaml" );
        /** N.B. Although readdirSync() now offers recursive: true as an option, the implementation of the path in
         *  the returned Dirent objects is not stable in current releases, with "entry.path" being deprecated in
         *  favor of "entry.parentPath"... but my current environment using 20.9.0 does not give path parentPath at
         *  all, and still hands back path. The most stable/predictable choice is to not implement recursive:true
         *  and do the recursion ourselves here.
         */
        for ( let entry of fs.readdirSync( td, { withFileTypes: true } ) ) {
            this.log.debug( 6, "%1 loading template defs, considering %2", this, entry.name );
            if ( entry.name.startsWith( "." ) ) {
                continue;
            }
            const pn = path.join( td, entry.name );
            if ( entry.isDirectory() ) {
                await this._load_template_dir( pn );
                continue;
            } else if ( ! entry.isFile() ) {
                continue;
            }
            try {
                if ( entry.name.endsWith( ".zip" ) ) {
                    this.log.debug( 5, "%1 loading template data from ZIP contents in %2", this, pn );
                    const StreamZip = require( 'node-stream-zip' );
                    const zip = new StreamZip.async( { file: pn } );
                    const entries = await zip.entries();
                    for ( const ze of Object.values( entries ) ) {
                        if ( ze.isDirectory ) {
                            continue;
                        }
                        try {
                            let d;
                            if ( ze.name.endsWith( ".yaml" ) || ze.name.endsWith( ".yml" ) ) {
                                this.log.debug( 5, "%1 loading template data from %2 in ZIP %3", this, ze.name, pn );
                                const buf = await zip.entryData( ze.name );
                                d = yaml.load( buf.toString( 'utf-8' ), { filename: ze.name } );
                            } else if ( ze.name.endsWith( ".json" ) ) {
                                this.log.debug( 5, "%1 loading template data from %2 in ZIP %3", this, ze.name, pn );
                                const buf = await zip.entryData( ze.name );
                                d = JSON.parse( buf.toString( 'utf-8' ) );
                            } else {
                                this.log.debug( 5, "%1 zip file %2 entry %3 ignored: unrecognized/unusable file type",
                                    this, pn, ze.name );
                                continue;
                            }
                            this.save_template_data( d, pn, ze.name );
                        } catch ( err ) {
                            this.log.err( "%1 file %2 in %3 could not be loaded: %4", this, ze.name, pn, err );
                            this.sendError( "File {0:q} in {1:q} is unreadable; please see the log file for details.",
                                ze.name, pn );
                        }
                    }
                    zip.close();
                } else if ( entry.name.endsWith( '.yaml' ) || entry.name.endsWith( '.yml' ) ) {
                    this.log.debug( 5, "%1 loading YAML template data from %2", this, pn );
                    try {
                        const buf = fs.readFileSync( pn );
                        let d = yaml.load( buf.toString( 'utf-8' ), { filename: entry.name } );
                        this.save_template_data( d, path.dirname( pn ), path.basename( pn ) );
                    } catch ( err ) {
                        this.log.err( "%1 can't read/load templates from %2: %3", this, pn, err );
                        this.sendError( "File {0:q} is unreadable; please see the log file for details.", pn );
                        this.log.exception( err );
                    }
                } else if ( entry.name.endsWith( '.json' ) ) {
                    this.log.debug( 5, "%1 loading JSON template data from %2", this, pn );
                    try {
                        const buf = fs.readFileSync( pn );
                        let d = JSON.parse( buf.toString( 'utf-8' ) );
                        this.save_template_data( d, path.dirname( pn ), path.basename( pn ) );
                    } catch ( err ) {
                        this.log.err( "%1 can't read/load templates from %2: %3", this, pn, err );
                        this.sendError( "File {0:q} is unreadable; please see the log file for details.", pn );
                        this.log.exception( err );
                    }
                } else {
                    this.log.debug( 5, "%1 file %2 ignored: unrecognized/unusable file type", this, pn );
                }
            } catch ( err ) {
                this.log.err( "%1 can't read/load template defs from %2: %3", this, pn, err );
                this.sendError( "File {0:q} is unreadable; please see the log file for details.", pn );
            }
        }
    }

    /** Load implementation data from data files. */
    async _load_implementation() {
        impl = await this.loadBaseImplementationData( 'mqtt', __dirname );
        impl.templates = impl.templates || {};

        const confdir = Configuration.getConfig( "reactor.confdir" );
        let r = util.loadData( [ "." ], "local_mqtt_devices", confdir );
        if ( r ) {
            this.log.logp( 8, this, '_load_implmap', "local templates %1", Object.keys( r.templates || {} ) );
            impl.templates = util.combine( impl.templates || {}, r.templates || {} );
            r = undefined;
        }

        const distdir = path.resolve( __dirname, "templates" );
        await this._load_template_dir( distdir );

        const td = path.resolve( confdir, "mqtt_templates" );
        await this._load_template_dir( td );

        // Register local Capabilities
        for ( const [n,cap] of localCapabilities.entries() ) {
            this.log.debug( 5, "%1 registering template capability %2 with system capabilities", this, n );
            Capabilities.loadCapabilityData( { [n]: cap } );
        }

        // Freeze templates
        Object.freeze( impl.templates );

        /* Hash the implementation data and controller version; for entity config change detection. */
        impl._ctrl_version = version;
        impl._sys_cap_info = Capabilities.getSysInfo();

        let hash = util.hash( JSON.stringify( impl ) );
        impl.hash = hash;
}

    async do_entity_inits( entity ) {
        let eid = entity.getID();
        let ec = this.config.entities[ eid ];
        let tpl = entity.__template;
        let q = [];
        if ( isUndef( tpl.init ) ) {
            /* Nothing, but fall through so that query can be sent, if set */
        } else if ( "string" === typeof tpl.init ) {
            q.push( { topic: tpl.init } );
        } else if ( Array.isArray( tpl.init ) ) {
            tpl.init.forEach( el => {
                if ( "string" === typeof el ) {
                    q.push( { topic: el } );
                } else if ( null !== el && "object" === typeof el && "string" === typeof el.topic ) {
                    q.push( el );
                } else {
                    this.log.err( '%1 ignoring init for %2 because it cannot be parsed: (%3) %4', this, eid,
                        typeof el, el );
                    this.sendError( "Ignoring init configuration for {0:q} because it is malformed.", eid );
                    return;
                }
            });
        } else {
            this.log.err( '%1 ignoring invalid "init" data for %2: (%3) %4', this, entity, typeof tpl.init, tpl.init );
            this.sendError( "Ignoring init configuration for {0:q} because it is malformed.", eid );
            return;
        }
        this.log.debug( 5, "%1 sending %2 inits for %3", this, q.length, eid );
        let promises = [];
        for ( let el of q ) {
            const s = (el.topic || "").replace( /%([^%]+)%/g, ( m, p ) => {
                let altstr = "";
                const k = p.indexOf( ':' );
                if ( k >= 0 ) {
                    altstr = p.substring( k + 1 );
                    p = p.substring( 0, k );
                }
                return String( coalesce( ec[p], altstr ) );
            });
            this.log.debug( 6, "%1 init for %2, publishing %3 payload %4", this, eid, s, el.payload );
            let payload = el.payload;
            if ( ! isUndef( payload ) ) {
                if ( 'json' === el.type || ( isUndef( el.type ) && "object" === typeof payload ) ) {
                    payload = JSON.stringify( payload );
                } else {
                    payload = String( payload );
                }
            }
            let opts = {
                qos: coalesce( parseInt( el.qos ), this.default_qos ),
                retain: parseBoolean( el.retain )
            };
            const p = this._publish( s, payload, opts );
            promises.push( p );
        }
        return Promise.allSettled( promises ).then( () => {
            this.log.debug( 6, "%1 all inits for %2 settled", this, eid );
            if ( tpl.query ) {
                entity.perform( 'x_mqtt_device.poll', {} );
            }
        });
    }

    /* Send any inits for configured devices. */
    async do_inits() {
        for ( let entity of this.inits ) {
            try {
                await this.do_entity_inits( entity );
            } catch( err ) {
                this.log.err( "%1 failed to init %2: %3", this, entity, err );
                this.log.exception( err );
            }
        }
    }

    handle_event( eid, event, topic, payload, packet ) {
        this.log.debug( 5, "%1 handling %2 (event %3) for %4: %5", this, topic, event, eid, payload );
        let e = this.findEntity( eid );
        if ( ! e ) {
            this.log.err( "%1 can't handle %2 for %3: entity does not exist", this, topic, eid );
            return;
        }
        let ev = e.__events[ event ];
        if ( ! ev ) {
            this.log.err( "%1 missing event %2 on %3", this, event, eid );
            return;
        }
        if ( "function" === typeof( ev ) ) {
            ev( e, topic, payload, packet );
            return;
        }
        if ( this.log.getEffectiveLevel() >= 5 ) {
            this.log.debug( 5, "%1 topic %2 affects entity %3 attributes %4", this, topic, eid,
                ev.map( el => el.attribute ).join(', ') );
        }
        let ctx = this._get_lexp_context( e.__template );  /* template items are in global scope -- DEPRECATE THIS? */
        // lexp.define_var( ctx, 'entity', e.extract() ); // handled below so each attribute/expr has updated values for entity
        lexp.define_var( ctx, 'topic', topic );
        lexp.define_var( ctx, 'config', e.__template );  /* This is better */
        lexp.define_var( ctx, 'system', this.system.extract() );
        lexp.define_var( ctx, 'packet', packet || {} );
        this.log.debug( 6, "%1 template in context is %2", this, e.__template );
        e.deferNotifies( true );
        try {
            if ( topic !== e.__template?.lwt ) {
                /* Any topic other than LWT marks device online. Do this early, so event can change it if necessary. */
                e.setAttribute( 'x_mqtt_device.online', true );
            }
            for ( const d of ev.values() ) {
                lexp.define_var( ctx, 'entity', e.extract() );  // N.B. fresh copy for each attribute!
                const attr = d.attribute;
                let newval = null;
                this.log.debug( 6, "%1 entity %2 attribute %3 source %4", this, eid, attr, d);
                if ( true === d.json_payload ) {
                    try {
                        lexp.define_var( ctx, 'payload', JSON.parse( payload ) );
                    } catch ( err ) {
                        this.log.err( "%1 entity %2 received topic %3 unparsable JSON payload: %4", this, e, topic,
                            err );
                        this.log.err( "%1 payload (length %3): %2", this, payload, (payload||"").length );
                        lexp.define_var( ctx, 'payload', null );
                        e.setAttribute( attr, null );
                        continue;
                    }
                } else {
                    lexp.define_var( ctx, 'payload', String( payload ) );
                }
                if ( ! isUndef( d.value ) ) {
                    newval = d.value;
                } else if ( d.source && true === d.json_payload ) {
                    newval = util.deref( lexp.get_var( ctx, 'payload' ), d.source, null );
                } else {
                    newval = coalesce( lexp.get_var( ctx, 'payload' ) );  // default value is the entire payload
                }
                if ( d.if_expr ) {
                    /* if_expr: conditional expression to determine if this attribute should be modified. This is
                    *  typically used to determine if the MESSAGE contains necessary data.
                    */
                    let ce = expr_ce[ d.if_expr ];
                    if ( isUndef( ce ) ) {
                        try {
                            ce = expr_ce[ d.if_expr ] = lexp.compile( d.if_expr );
                        } catch ( err ) {
                            this.log.err( "%1 can't compile if_expr %2: %3 (%4 %5)", this, d.if_expr, err, eid, topic );
                            ce = expr_ce[ d.if_expr ] = false;
                        }
                    }
                    if ( ce ) {
                        try {
                            let res = lexp.run( ce, ctx );
                            if ( "boolean" !== typeof res ) {
                                this.log.warn( "%1 if_expr expression %2 returned non-boolean (%3)", this, d.if_expr,
                                    typeof res );
                            }
                            if ( ! res ) {
                                continue; /* if_expr returns false; skip this attribute */
                            }
                        } catch ( err ) {
                            this.log.err( "%1 if_expr expression %2 failed evaluation: %3", this, d.if_expr, err );
                            this.log.err( "%1 payload: %2", this, payload );
                            continue; /* hard error evaluating is always false */
                        }
                    } else {
                        continue; /* hard error compiling if_expr is always false */
                    }
                }
                if ( d.expr ) {
                    let ce = expr_ce[ d.expr ];
                    if ( isUndef( ce ) ) {
                        try {
                            lexp.define_var( ctx, 'value', newval );
                            ce = expr_ce[ d.expr ] = lexp.compile( d.expr );
                        } catch ( err ) {
                            this.log.err( "%1 can't compile expression %2: %3 (%4 %5)", this, d.expr, err, eid, topic );
                            ce = expr_ce[ d.expr ] = false;
                        }
                    }
                    if ( ce ) {
                        /* Make current value available like in other Controllers? So far, no need. */
                        try {
                            newval = lexp.run( ce, ctx );
                        } catch ( err ) {
                            this.log.err( "%1 expression %2 failed evaluation: %3", this, d.expr, err );
                            this.log.err( "%1 payload: %2", this, payload );
                            newval = null;
                        }
                    }
                }
                if ( null !== d.map && "object" === typeof d.map ) {
                    if ( isUndef( d.map[ newval ] ) ) {
                        // Leave newval untouched unless map_unmatched is specified.
                        if ( ! isUndef( d.map_unmatched ) ) {
                            newval = d.map_unmatched;
                        }
                    } else {
                        newval = d.map[ newval ];
                    }
                }
                this.log.debug( 5, "%1 topic %2 setting %3 %4=%5", this, topic, eid, attr, newval );
                e.setAttribute( attr, coalesce( newval ) );  // coalesce to ensure NaN or undefined -> null
            }
        } finally {
            e.deferNotifies( false );
        }
    }

    handle_lwt( e, topic, payload, packet ) {  // eslint-disable-line no-unused-vars
        const eid = e.getID();
        this.log.notice( "%1 received LWT (%2) for %3", this, topic, eid );
        e.setAttribute( 'x_mqtt_device.online', false );
    }

    match_entity( patterns, eid ) {
        for ( let p of ( patterns || [] ) ) {
            if ( p instanceof RegExp ) {
                if ( ! p.test( eid ) ) {
                    continue;
                }
            } else if ( p.startsWith( '/' ) ) {
                /* Regex */
                const k = p.lastIndexOf( '/' );
                const fl = p.substring( k + 1 );
                try {
                    const re = new RegExp( p.substring( 1, k ), fl );
                    if ( ! re.test( eid ) ) {
                        continue;
                    }
                } catch ( err ) {
                    this.log.err( "%1 failed regular expression %2: %3", this, p, err );
                }
            } else {
                let m = p.split( '>' );
                let n = eid.split( '>' );
                if ( '*' !== m[0] && m[0] !== n[0] ) {
                    continue;
                }
                if ( '*' !== m[1] && m[1] !== n[1] ) {
                    continue;
                }
            }
            return p;
        }
        return false;
    }

    match_capability( patterns, cap ) {
        for ( let p of ( patterns || [] ) ) {
            if ( p instanceof RegExp ) {
                if ( ! p.test( cap ) ) {
                    continue;
                }
            } else if ( p.startsWith( '/' ) ) {
                /* Regex */
                const k = p.lastIndexOf( '/' );
                const fl = p.substring( k + 1 );
                try {
                    const re = new RegExp( p.substring( 1, k ), fl );
                    if ( ! re.test( cap ) ) {
                        continue;
                    }
                } catch ( err ) {
                    this.log.err( "%1 failed regular expression %2: %3", this, p, err );
                }
            } else if ( p !== cap ) {
                continue;
            }
            return p;
        }
        return false;
    }

    match_group( groups, e ) {
        let struct = this.getStructure();
        for ( let gid of ( groups || [] ) ) {
            let group = struct.findEntity( gid );
            if ( group && group.hasMember( e ) ) {
                return true;
            }
        }
        return false;
    }

    subscribe_entities() {
        let struct = this.getStructure();
        this.log.notice( "%1 subscribing to echo entities", this );
        const config = this.config.echo;
        let count = 0;
        for ( let cid of Object.keys( struct.getControllers() ) ) {
            let controller = struct.getControllerByID( cid );
            if ( controller === this ) {
                continue; /* don't self-subscribe */
            }
            let entities = controller.getEntities();
            for ( let [ eid, entity ] of Object.entries( entities ) ) {
                const canonid = entity.getCanonicalID();
                if ( config.include_entities && ! this.match_entity( config.include_entities, canonid ) ) {
                    this.log.debug( 7, "%1 entity %2 not on include_entities list; skipped", this, canonid );
                    continue;
                }
                if ( config.exclude_entities && this.match_entity( config.exclude_entities, canonid ) ) {
                    this.log.debug( 7, "%1 entity %2 is on exclude_entities list; skipped", this, canonid );
                    continue;
                }
                this.log.debug( 6, "%1 echo function: subscribing to %2", this, eid );
                this.subscribe( entity );
                ++count;
            }
        }
        this.log.info( "%1 subscribed to %2 entities for echo", this, count );
    }

    notify_mqtt( event ) {
        if ( ! ( this.config.echo && this.client ) ) {
            return;
        }
        const entity = event.sender;
        const controller = entity.getController();
        if ( this === controller ) {
            /* No self-echo */
            return;
        }
        this.log.debug( 6, "%1 echo handling entity notification for %2", this, entity );
        this.echo_entity( entity );
    }

    echo_entity( entity ) {
        this.log.debug( 6, "%1 echo publishing capabilities for entity %2", this, entity );
        let config = this.config.echo;
        const cid = entity.getController().getID();
        const eid = entity.getID();

        /* Check groups, which may be dynamic. */
        if ( config.include_groups && ! this.match_group( config.include_groups ) ) {
            this.log.debug( 7, "%1 entity %2 not in include_groups list; skipped", this, eid );
            return;
        }
        if ( config.exclude_groups && this.match_group( config.exclude_groups ) ) {
            this.log.debug( 7, "%1 entity %2 is in exclude_groups group; skipped", this, eid );
            return;
        }

        let etop = eid;
        if ( "name" === config.entity_identifier ) {
            etop = entity.getName().replace( /(\/|\p{C})/gu, "?" );  /* Remove / and Unicode control chars */
        } else if ( "combined" === config.entity_identifier || "id/name" === config.entity_identifier ) {
            etop = `${eid}/${entity.getName().replace( /(\/|\p{C})/gu, "?" )}`;
        } else if ( "name/id" === config.entity_identifier ) {
            etop = `${entity.getName().replace( /(\/|\p{C})/gu, "?" )}/${eid}`;
        } else if ( "id" !== ( config.entity_identifier || "id" ) ) {
            this.log.warn( "%1 configuration entity_identifier unrecognized value: %2", this, config.entity_identifier );
        }

        /* Check capabilities */
        /* ??? extend syntax to include attribute, so specific attributes can be included or excluded? */
        let capabilities = entity.getCapabilities();
        let sent = false;
        for ( let cap of capabilities ) {
            if ( config.include_capabilities && ! this.match_capability( config.include_capabilities, cap ) ) {
                /* Not on list */
                this.log.debug( 6, "%1 echo cap %2 rejected -- not included", this, cap );
                continue;
            }
            if ( config.exclude_capabilities ) {
                if ( this.match_capability( config.exclude_capabilities, cap ) ) {
                    /* On exclude list */
                    this.log.debug( 6, "%1 echo cap %2 rejected -- excluded", this, cap );
                    continue;
                }
            } else if ( this.match_capability( [ /^x_/ ], cap ) ) {
                this.log.debug( 6, "%1 echo cap %2 rejected -- default reject extended %2", this, cap );
                continue;
            }

            /* Build payload for this capability and send data */
            let payload = {};
            const attrs = entity.enumAttributes( cap );
            if ( attrs.length > 0 ) {
                for ( let attr of attrs ) {
                    let an = attr.split( /\./ ).pop();
                    try {
                        payload[an] = entity.getAttribute( attr );
                    } catch ( err ) {  // eslint-disable-line no-unused-vars
                        payload[an] = null;
                    }
                }
                payload = JSON.stringify( payload );
                const topic = `${this.topic_prefix}${cid}/${etop}/state/${cap}`;
                this.log.debug( 6, "%1 echo publishing %2 %3", this, topic, payload );
                this._publish( topic, payload, { qos: this.echo_qos, retain: this.echo_retain } ).catch( err => {} );  // eslint-disable-line no-unused-vars
                sent = true;
            }
        }
        if ( false !== config.primary_attribute ) {
            const topic = `${this.topic_prefix}${cid}/${etop}/value`;
            const payload = JSON.stringify( coalesce( entity.getPrimaryValue() ) );
            this.log.debug( 6, "%1 echo publishing primary attribute topic %2 %3", this, topic, payload );
            this._publish( topic, payload, { qos: this.echo_qos, retain: this.echo_retain } ).catch( err => {} );  // eslint-disable-line no-unused-vars
            sent = true;
        }
        if ( ! sent ) {
            /* Nothing sent; no capabilities included or all excluded. Unsubscribe from entity. */
            this.log.notice( "%1 removing %2 from echo watch because it has no publishable capabilities", this, entity );
            this.unsubscribe( entity );
        }
    }

    publish_alerts() {
        if ( this.client && this.isReady() ) {
            const alerts = AlertManager.getExportData();  /* Safe form/content */
            const topic = `${this.topic_prefix}alerts`;
            this.log.debug( 6, "%1 publishing alerts topic %2 %3", this, topic, alerts );
            this._publish( topic, JSON.stringify( alerts ),
                { qos: this.echo_qos, retain: this.echo_retain }
            ).catch( err => {} );  // eslint-disable-line no-unused-vars
        }
    }

    notify( event ) {
        try {
            if ( event.sender === event.currentSender ) {  // only send for direct notifications
                switch ( event.type ) {
                    case 'entity-changed':
                        this.notify_mqtt( event );
                        break;
                    case 'alerts-changed':
                        this.publish_alerts( event.sender );
                        break;
                    default:
                        /* ignored */
                }
            }
        } catch ( err ) {
            this.log.err( "%1 exception while handling %2 event from %3: %4", this, event.type, event.sender, err );
            this.log.exception( err );
        }

        return super.notify( event );  /* REQUIRED! */
    }

    /* Handle a controller event for an echo device. Incoming 'perform' messages are translated into
    *  entity and action, with the message payload being the action arguments. This allows pretty
    *  much any action in any capability known to any entity to be performed via MQTT.
    */
    handle_controller_event( topic, payload, packet ) {  // eslint-disable-line no-unused-vars
        if ( Buffer.isBuffer( payload ) ) {
            payload = payload.toString( 'UTF-8' );
        } else if ( null === payload ) {
            payload = "";
        }
        let m = topic.split( '/' );
        if ( m.length < 3 || m[1] !== this.my_ident ) {
            return;  /* Not for me! */
        }
        this.log.debug( 5, "%1 handling incoming %2 %3", this, topic, payload );
        if ( "ping" === m[2] ) {
            /* reactor/mqtt/ping */
            this._publish( `${this.topic_prefix}pong`, payload || Date.now(), { qos: this.default_qos, retain: false } )
                .catch( err => {} );  // eslint-disable-line no-unused-vars
            return;
        } else if ( "Rule" === m[2] ) {
            if ( "state" === m[4] ) {
                return;
            } else if ( "query" === m[4] ) {
                /* reactor/mqtt/Rule/rule-j29z3yt/query */
                const Rule = require( "server/lib/Rule" );
                if ( Rule.exists( m[3] ) ) {
                    try {
                        this.publish_rule( m[3], {} );
                    } catch ( err ) {
                        this.log.exception( err );
                    }
                } else {
                    this.log.warn( "%1 topic %2: rule does not exist", this, topic );
                }
            } else {
                this.log.warn( "%1 invalid topic %2 (ignored)", this, topic );
            }
            return;
        } else if ( "Reaction" === m[2] ) {
            /* reactor/mqtt/Reaction/reaction_name_or_id/run (payload ignored) */
            /* reactor/mqtt/Reaction/reaction_name_or_id/stop (payload ignored) */
            /* reactor/mqtt/Reaction/reaction_name_or_id/query (payload ignored) */
            const rid = m[3];
            const GlobalReaction = require( "server/lib/GlobalReaction" );
            const reactions = GlobalReaction.getReactions();
            const mrid = rid.toLocaleLowerCase();
            let reaction;
            for ( let r of reactions ) {
                const rr = GlobalReaction.getInstance( r );
                if ( rid === rr.getID() ) {
                    reaction = rr;
                    break;
                }
                if ( mrid === rr.getName().toLocaleLowerCase() ) {
                    reaction = rr;
                    break;
                }
            }
            if ( ! reaction ) {
                this.log.err( "%1 topic %2 fails; reaction %3 not found", this, topic, rid );
                return;
            }
            const cmd = (m[4] || "query").toLowerCase();
            const engine = require( "server/lib/Engine" ).getInstance();
            if ( "stop" === cmd ) {
                engine.stopReaction( reaction.getID(), null, "MQTT Topic" );
                return;
            } else if ( "run" === cmd ) {
                engine.queueReaction( reaction.getID(), null, true );
                return;
            } else if ( "query" === cmd ) {
                const ptopic = `${this.topic_prefix}Reaction/${reaction.getID()}/status`;
                const payload = {
                    id: reaction.getID(),
                    name: reaction.getName(),
                    enabled: !reaction.disabled
                };
                try {
                    payload.queued = engine.isReactionQueued( reaction.getID() );
                    payload.running = engine.isReactionRunning( reaction.getID() );
                } catch ( err ) {  // eslint-disable-line no-unused-vars
                    this.log.err( "%1 Unable to query Engine for reaction status; please make sure you are running Reactor 25111 or higher.", this );
                    payload.queued = null;
                    payload.running = null;
                }
                this.log.debug( 5, "%1 publishing reaction status %2 %3", this, ptopic, payload );
                this._publish( ptopic, payload, { qos: 0, retain: false } ).catch( err => {} );  // eslint-disable-line no-unused-vars
                return;
            } else {
                this.log.err( "%1 topic %2 invalid command %3", this, topic, m[4] );
                return;
            }
        } else if ( "Expr" === m[2] ) {
            /* reactor/mqtt/Expr/somevarname/query (payload ignored) */
            /* reactor/mqtt/Expr/somevarname/set payload=value */
            if ( "value" === m[4] ) {
                return;
            } else if ( "query" === m[4] ) {
                /* Echo value */
                this.publish_global_var( m[3], {} );
            } else if ( "set" === m[4] ) {
                const Engine = require( "server/lib/Engine" );
                /* Set value */
                let value = null;
                try {
                    value = coalesce( JSON.parse( payload ) );
                } catch ( err ) {  // eslint-disable-line no-unused-vars
                    this.log.err( "%1 unparsable JSON payload in %2 %3", this, topic, payload );
                }
                try {
                    this.log.debug( 5, "%1 handling %2 set %3=%4", this, topic, m[3], value);
                    // Engine.getInstance().setGlobalVariable( m[3], value );  /* requires Engine 22029 */
                    Engine.getInstance().setVariable( m[3], false, value, false, false );
                } catch ( err ) {
                    this.log.err( "%1 topic %2 can't set global variable %3: %4", this, topic, m[3], err );
                    this.log.exception( err );
                }
            } else {
                this.log.warn( "%1 invalid topic %2 (ignored)", this, topic );
            }
            return;
        } else if ( 5 === m.length && "query" === m[4] ) {
            /* reactor/mqtt/vera/device_10/query */
            let eid = m[2] + '>' + m[3];
            const entity = this.getStructure().findEntity( eid );
            if ( entity ) {
                this.echo_entity( entity );
            } else {
                this.log.warn( "%1 received topic %2 for entity %3, no such entity (ignored)", this, topic, eid );
                return;
            }
        }
        /* Ex: reactor/my_ident/vera/device_10/perform/power_switch/on -- 7 parts */
        if ( m.length < 5 || "perform" !== m[4] ) {
            /* Not something we're handling here. The "state" and "value" msg echoes are also caught here. */
            this.log.debug( 7, "%1 quietly ignoring %2", this, topic );
            return;
        }
        let eid = `${m[2]}>${m[3]}`;
        let entity = this.getStructure().findEntity( eid );
        if ( ! entity ) {
            this.log.notice( "%1 received topic %2 for entity %3, no such entity (ignored)", this, topic, eid );
            return;
        }
        if ( ! entity.hasCapability( m[5] ) ) {
            this.log.notice( "%1 topic %2 for entity %3: entity does not support capability %4", this, topic, eid, m[5] );
            return;
        }
        let action = m[5] + '.' + m[6];
        let parameters = {};
        if ( "undefined" !== typeof payload && null !== payload && "" !== payload ) {
            try {
                parameters = JSON.parse( payload );
            } catch ( err ) {
                this.log.warn( "%1 echo topic %2 invalid JSON: %3", this, topic, err );
                this.log.info( "payload=%1", payload );
                return;
            }
        }
        this.log.debug( 5, "%1 echo sending action %2 parameters %3", this, action, parameters );
        entity.perform( action, parameters ).then( data => {
            this.log.info( "%1 action %2 on %3 completed (MQTT echo)", this, action, entity );
            return data;
        });
    }

    performOnEntity( entity, actionName, params ) {
        this.log.debug( 5, "%1 perform %2 on %3 with %4", this, actionName, entity, params );
        const eid = entity.getID();
        const cf = this.config.entities[ eid ] || {};
        let act = ( entity.__actions || {} )[ actionName ];
        if ( false === act ) {
            return Promise.reject( new Error( `Action ${actionName} is not supported by this entity.` ) );
        } else if ( ! act || true === act ) {
            return super.performOnEntity( entity, actionName, params );
        } else if ( "string" === typeof act ) {
            act = { topic: act };
        } else if ( "object" !== typeof act || "string" !== typeof act?.topic ) {
            return Promise.reject( new Error( `Invalid action implementation for ${actionName}` ) );
        }

        let ctx = this._get_lexp_context();
        lexp.define_var( ctx, 'entity', entity.extract() );
        lexp.define_var( ctx, 'system', this.system.extract() );
        lexp.define_var( ctx, 'parameters', params || {} );
        lexp.define_var( ctx, 'config', cf );
        for ( const [key, val] of Object.entries( cf ) ) {
            lexp.define_var( ctx, key, val );
        }

        let topic = act.topic.replace( /%([^%]+)%/g, ( m, p ) => {
            let altstr = "";
            const k = p.indexOf( ':' );
            if ( k >= 0 ) {
                altstr = p.substring( k + 1 );
                p = p.substring( 0, k );
            }
            return String( coalesce( cf[p], altstr ) );
        });

        let payload;
        if ( null === act.payload || isUndef( act.payload ) ) {
            /* Nada -- payload undefined */
        } else if ( "object" === typeof act.payload ) {
            payload = coalesce( act.payload.value );
            if ( act.payload.parameter ) {
                payload = util.deref( params, act.payload.parameter );
            }
            if ( act.payload.expr ) {
                if ( isUndef( expr_ce[ act.payload.expr ] ) ) {
                    try {
                        expr_ce[ act.payload.expr ] = lexp.compile( act.payload.expr );
                    } catch ( err ) {
                        this.log.err( "%1 entity %2 action %3 payload expression failed: %4", this, eid, actionName, err );
                        expr_ce[ act.payload.expr ] = false;
                    }
                }
                if ( expr_ce[ act.payload.expr ] ) {
                    try {
                        lexp.define_var( ctx, "value", coalesce( payload ) );
                        payload = lexp.run( expr_ce[ act.payload.expr ], ctx );
                    } catch ( err ) {
                        this.log.err( "%1 entity %2 action %3 payload expression failed: %4", this, eid, actionName, err );
                    }
                }
            }
            payload = coalesce( payload );  // null, undefined, NaN => null
            if ( 'json' === act.payload.type ) {
                payload = JSON.stringify( payload );
            } else {
                if ( 'raw' !== ( act.payload.type || 'raw' ) ) {
                    this.log.warn( "%1 entity %2 action %3 unsupported payload type %4",
                        this, eid, actionName, act.payload.type );
                }
                payload = String( payload );
            }
        } else {
            payload = String( act.payload );
        }

        let opts = {
            qos: coalesce( parseInt( act.qos ), this.default_qos ),
            retain: parseBoolean( act.retain )
        };

        this.log.info( "%1 perform %2 on %3: publishing %4 %5", this, actionName, eid, topic, payload );
        return this._publish( topic, payload, opts );
    }

    subscribe_global_vars() {
        const GlobalExpression = require( 'server/lib/GlobalExpression' );
        this.log.info( "%1 subscribing to global variables", this );
        const exps = GlobalExpression.getExpressions();
        for ( let id of exps ) {
            this.log.debug( 5, "%1 subscribing to %2", this, id );
            const exp = GlobalExpression.getInstance( id );
            this.subscribe( exp, this.publish_global_var.bind( this, id ) );
            try {
                this.publish_global_var( id );
            } catch ( err ) {
                this.log.warn( "%1 error while publishing current global %2 value: %3", this, exp.name, err );
            }
        }
    }

    publish_global_var( id ) {
        const GlobalExpression = require( 'server/lib/GlobalExpression' );
        if ( GlobalExpression.exists( id ) ) {
            const exp = GlobalExpression.getInstance( id );
            const payload = JSON.stringify( exp.getValue() );
            const topic = `${this.topic_prefix}Expr/${exp.name}/value`;
            const opts = {
                qos: this.echo_qos,
                retain: this.echo_retain
            };
            this.log.debug( 5, "%1 publishing gvar %2 value %3 %4", this, exp.name, topic, payload.substring( 0, 63 ) );
            this._publish( topic, payload, opts ).catch( err => {} );  // eslint-disable-line no-unused-vars
        } else {
            this.log.warn( "%1 can't publish global expr %2: does not exist", this, id );
        }
    }

    subscribe_rules() {
        const Engine = require( "server/lib/Engine" );
        const engine = Engine.getInstance( this.getStructure() );
        const Rulesets = require( "server/lib/Rulesets" );
        const Ruleset = require( "server/lib/Ruleset" );
        const Rule = require( "server/lib/Rule" );
        const rulesets = Rulesets.getInstance().getRulesets();
        for ( let rsid of rulesets ) {
            const ruleset = Ruleset.getInstance( rsid );
            for ( let rid of ruleset.getRules() ) {
                if ( Rule.exists( rid ) ) {
                    const rule = Rule.getInstance( rid, engine );
                    const bound_handler = this.publish_rule.bind( this, rid );
                    this.subscribe( rule, bound_handler );
                    this.subscribe( rule.getRuleStates(), bound_handler );
                    try {
                        this.publish_rule( rid, { type: 'rule-state' } );
                    } catch ( err ) {
                        this.log.err( "%1 failed initial publish of %2: %3", this, rid, err );
                    }
                }
            }
        }
    }

    publish_rule( rid, event ) {
        this.log.debug( 6, "%1 rule %2 event %3", this, rid, event.type );
        if ( "rule-state" !== event.type ) {
            return;
        }
        const Rule = require( "server/lib/Rule" );
        if ( Rule.exists( rid ) ) {
            const Engine = require( "server/lib/Engine" );
            const engine = Engine.getInstance( this.getStructure() );
            const rule = Rule.getInstance( rid, engine );
            const payload = JSON.stringify( coalesce( rule.getConditionState( 'rule' ).evalstate ) );
            const topic = `${this.topic_prefix}Rule/${rule.getID()}/state`;
            const opts = {
                qos: this.echo_qos,
                retain: this.echo_retain
            };
            this.log.debug( 5, "%1 publishing rule %2 state %3 %4", this, rule, topic, payload );
            this._publish( topic, payload, opts ).catch( err => {} );  // eslint-disable-line no-unused-vars
        } else {
            this.log.warn( "%1 can't publish %2: does not exist", this, rid );
        }
    }

    /** ---------------- EXTERNAL INTERFACES -------------------- */

    /** Allow another controller or plugin to subscribe to MQTT topics.
     *  The callback will be used to respond to a matching topic.
     *  It may be a function, which is called with the topic and payload
     *  as arguments, or a Controller, which is sent a notification of
     *  type 'ext-mqtt-topic' with the topic and payload in data.
     */
    extSubscribeTopic( obj, topic, callback, ...args ) {
        if ( ! ( obj instanceof Controller ) || this.isEqual( obj ) ) {
            /* Must be controller, and don't allow mistaken pass of MQTTController */
            throw new TypeError( "Invalid first argument, must be Controller (this)" );
        }
        if ( "string" !== typeof topic ) {
            throw new TypeError( "Invalid topic" );
        }
        if ( ! ( isUndef( callback ) || "function" === typeof callback ) ) {
            throw new TypeError( "Invalid callback" );
        }
        let pattern = '^' + topic.replace( /[.?*]/g, "\\$&" ) + '$';
        pattern = pattern.replace( /\/\+/g, "/[^/]+" );
        pattern = pattern.replace( /\/#/g, "/.*" );
        let re = new RegExp( pattern );
        this.ext_patterns.push( { topic: topic, obj: obj, re: re, callback: callback, args: args } );
        this.log.debug( 5, "%1 extSubscribeTopic added %2 as %3 for %4 cb=%5", this, topic, re.toString(), obj,
            typeof callback );
    }

    /** Unsubscribe from a previously-subscribed topic. */
    extUnsubscribeTopic( obj, topic ) {
        this.log.debug( "%1 extUnsubscribe %2 for %3", this, topic, obj );
        while ( true ) {
            const ix = this.ext_patterns.findIndex( el => el.obj.isEqual( obj ) && ( !topic || el.topic === topic ) );
            if ( ix < 0 ) {
                break;
            }
            this.log.debug( 5, "%1 extUnsubscribeTopic removing %2 for %3", this, this.ext_patterns[ix].topic,
                this.ext_patterns[ix].obj );
            this.ext_patterns.splice( ix, 1 );
        }
    }

    /** Publish a topic. */
    async extPublishTopic( topic, payload, opts ) {
        if ( "object" === typeof payload && ! opts.raw ) {
            payload = JSON.stringify( payload );
        } else if ( ! isUndef( payload ) ) {
            payload = String( payload );
        }
        opts = opts || {};
        opts.qos = coalesce( parseInt( opts.qos ), 0 );
        opts.retain = "boolean" === typeof opts.retain && opts.retain;
        this.log.debug( 5, "%1 extPublishTopic() publishing %2 qos %3 retain %4 payload %5", this,
            topic, opts.qos, opts.retain, payload || "" );
        return this._publish( topic, payload, opts );
    }

    /** Private method to dispatch topics to handlers */
    _handle_ext_event( topic, payload ) {
        let pt = this.ext_patterns.filter( el => topic.match( el.re ) );
        this.log.debug( 5, "%1 _handle_ext_event() %2 matches %3 subscriptions", this, topic, pt.length );
        for ( let p of pt ) {
            if ( "function" === typeof p.callback ) {
                try {
                    p.callback( topic, payload, ...(p.args || []) );
                } catch ( err ) {
                    this.log.err( "%1 error thrown by external topic %2 handler (callback): %3", this, topic, err );
                    this.log.exception( err );
                }
            } else if ( ! p.callback ) {
                const c = p.obj;
                try {
                    c.notify( {
                        type: 'ext-mqtt-topic',
                        sender: this,
                        currentSender: this,
                        data: { topic: topic, payload: payload, subscribed_as: p.topic },
                        args: p.args || []
                    });
                } catch ( err ) {
                    this.log.err( "%1 error thrown by external topic handler for %2 (in %3 notify()): %4", this,
                        topic, c, err );
                    this.log.exception( err );
                }
            }
        }
    }

    /** ---------------- POLLING --------------- */

    _start_polling( polled ) {
        const now = Date.now();
        this.poll_queue = [];
        if ( ! this.pollTimer ) {
            this.pollTimer = TimerBroker.getTimer( `mqtt-${this.id}-poller`, undefined,
                this._handle_polltimer.bind( this ) );
        } else {
            this.pollTimer.cancel();
        }
        for ( let eid of polled.values() ) {
            const e = this.findEntity( eid );
            const interval = Math.max( 1000, coalesce( parseInt( e.__template.poll_interval), 60 ) * 1000 );
            this.poll_queue.push( {
                entity_id: eid,
                interval: interval,
                next_poll: now + interval
            });
            this.log.debug( 5, "%1 poller scheduling %2 for interval %3ms", this, eid, interval );
        }
        this.last_poll = now;
        this._schedule_next_poll();
    }

    _schedule_next_poll() {
        if ( this.poll_queue.length > 0 && ! this.stopping ) {
            this.poll_queue.sort( (a,b) => a.next_poll - b.next_poll );
            const n = Math.max( this.last_poll + this.poll_frequency_ms, this.poll_queue[0].next_poll );
            this.log.debug( 5, "%1 poller next eligible %2 at %3, scheduling for %4", this,
                this.poll_queue[0].entity_id, this.poll_queue[0].next_poll, n );
            this.pollTimer.at( n );
        } else if ( this.pollTimer ) {
            this.log.notice( "%1 poller stopping; no eligible entities", this );
            this.pollTimer.cancel();
        }
    }

    _handle_polltimer() {
        const now = Date.now();
        while ( this.poll_queue.length > 0 ) {
            const el = this.poll_queue[ 0 ];
            if ( el.next_poll > now ) {
                break;
            }
            const entity = this.findEntity( el.entity_id );
            if ( entity ) {
                this.log.debug( 5, "%1 polling %2 (due %3)", this, el.entity_id, el.next_poll );
                entity.perform( 'x_mqtt_device.poll', {} );
                el.next_poll = now + el.interval;
                this.last_poll = now;
                break;
            } else {
                /* If entity is gone, remove from queue, try another. */
                this.log.notice( "%1 polling entity %2 no longer available, removing from queue", this, el.entity_id );
                this.poll_queue.splice( 0, 1 );
            }
        }
        this._schedule_next_poll();
    }

    /** ---------------- DEFAULT ACTION IMPLEMENTATIONS ------------------ */

    action_sys_system_restart( entity, params ) {  // eslint-disable-line no-unused-vars
        return new Promise( (resolve) => {
            this.log.info( "%1 performing sys_system.restart", this );
            this._recycle();
            resolve();
        });
    }

    action_x_mqtt_publish( entity, params ) {
        this.log.debug( 5, "%1 x_mqtt.publish action: %2", this, params );
        let payload = params.payload;
        if ( "" === payload || null === payload ) {
            payload = undefined;
        } else if ( "object" === typeof payload ) {
            payload = JSON.stringify( payload );
        }
        let opts = {
            qos: coalesce( parseInt( params.qos ), this.default_qos ),
            retain: parseBoolean( params.retain )
        };
        this.log.debug( 4, "%1 x_mqtt.publish publishing %2 payload %3", this, params.topic, payload );
        return this._publish( params.topic, payload, opts );
    }

    action_x_mqtt_device_poll( entity, params ) {  // eslint-disable-line no-unused-vars
        const tpl = entity.__template;
        const eid = entity.getID();
        const ec = this.config.entities[ eid ] || {};
        if ( tpl && tpl.query ) {
            let qq = "string" === typeof tpl.query ? { topic: tpl.query } : tpl.query;
            if ( ! ( qq && "object" === typeof qq && "string" === typeof qq.topic ) ) {
                return Promise.reject( new Error( `Invalid "query" config for ${eid}` ) );
            }
            const topic = qq.topic.replace( /%([^%]+)%/g, ( m, p ) => {
                let altstr = "";
                const k = p.indexOf( ':' );
                if ( k >= 0 ) {
                    altstr = p.substring( k + 1 );
                    p = p.substring( 0, k );
                }
                return String( coalesce( ec[p], altstr ) );
            });

            let payload;
            if ( null === qq.payload || isUndef( qq.payload ) ) {
                /* Nada -- payload undefined */
            } else if ( "object" === typeof qq.payload ) {
                payload = coalesce( qq.payload.value );
                if ( qq.payload.expr ) {
                    let ctx = this._get_lexp_context();
                    lexp.define_var( ctx, 'entity', entity.extract() );
                    lexp.define_var( ctx, 'system', this.system.extract() );
                    lexp.define_var( ctx, 'config', ec );
                    if ( isUndef( expr_ce[ qq.payload.expr ] ) ) {
                        try {
                            expr_ce[ qq.payload.expr ] = lexp.compile( qq.payload.expr );
                        } catch ( err ) {
                            this.log.err( "%1 entity %2 action x_mqtt_device.poll payload expression failed: %3", this, eid, err );
                            expr_ce[ qq.payload.expr ] = false;
                        }
                    }
                    if ( expr_ce[ qq.payload.expr ] ) {
                        try {
                            lexp.define_var( ctx, "value", coalesce( payload ) );
                            payload = lexp.run( expr_ce[ qq.payload.expr ], ctx );
                        } catch ( err ) {
                            this.log.err( "%1 entity %2 action x_mqtt_device.poll payload expression failed: %3", this, eid, err );
                        }
                    }
                }
                payload = coalesce( payload );  // null, undefined, NaN => null
                if ( 'json' === qq.payload.type ) {
                    payload = JSON.stringify( payload );
                } else {
                    if ( 'raw' !== ( qq.payload.type || 'raw' ) ) {
                        this.log.warn( "%1 entity %2 device query unsupported payload type %3",
                            this, eid, qq.payload.type );
                    }
                    payload = String( payload );
                }
            } else {
                payload = String( qq.payload );
            }

            let opts = {
                qos: coalesce( parseInt( qq.qos ), this.default_qos ),
                retain: parseBoolean( qq.retain )
            };

            this.log.info( "%1 x_mqtt_device.poll action sending %2 with %3 byte payload to %4", this, topic,
                payload ? payload.length : 0, eid );
            this.log.debug( 5, "%1 payload is %2", this, payload );
            return this._publish( topic, payload, opts );
        }
        this.log.warn("%1 unable to perform device poll: no 'query' defined for %2", this, entity);
        return Promise.reject( new Error( `No "query" configured for ${eid}` ) );
    }

    _get_lexp_context( ...args ) {
        const ctx = lexp.get_context( ...args );
        lexp.define_func_impl( ctx, 'hsltorgb', ( cx, h, s, l ) => MQTTController._hsl_to_rgb( h, s, l ) );
        lexp.define_func_impl( ctx, 'rgbtohsl', ( cx, r, g , b ) => MQTTController._rgb_to_hsl( r, g, b ) );
        return ctx;
    }

    /** Utility function to convert HSL (hue 0-360, sat 0-1, level 0-1) to RGB (0-255)
     *  Ref: https://www.rapidtables.com/convert/color/rgb-to-hsl.html
     *  Ref: https://en.wikipedia.org/wiki/HSL_and_HSV#HSV_to_RGB
     */
    static _hsl_to_rgb( hue, sat, lev ) {
        hue %= 360;
        sat = Math.min( 1, Math.max( 0, sat ) );
        lev = Math.min( 1, Math.max( 0, lev ) );
        let C = (1 - Math.abs( 2*lev - 1 )) * sat; // For HSV, use lev * sat
        let X = C * ( 1 - Math.abs( ( hue / 60 ) % 2 - 1 ) );
        let m = lev - C / 2;  // for HSV, use lev - C
        let rgb;
        if ( hue < 60 ) {
            rgb = [ C, X, 0 ];
        } else if ( hue < 120 ) {
            rgb = [ X, C, 0 ];
        } else if ( hue < 180 ) {
            rgb = [ 0, C, X ];
        } else if ( hue < 240 ) {
            rgb = [ 0, X, C ];
        } else if ( hue < 300 ) {
            rgb = [ X, 0, C ];
        } else {
            rgb = [ C, 0, X ];
        }
        for ( var k=0; k<3; ++k ) {
            rgb[k] = Math.max( 0, Math.min( 255, Math.floor( ( rgb[k] + m ) * 255 ) ) );
        }
        return rgb;
    }

    /** Utility function to convert RGB (0-255) to HSL (hue 0-360, sat 0-1, level 0-1) */
    static _rgb_to_hsl( r, g, b ) {
        let tol = 0.0001;
        r = Math.max( 0, Math.min( 255, r ) ) / 255;
        g = Math.max( 0, Math.min( 255, g ) ) / 255;
        b = Math.max( 0, Math.min( 255, b ) ) / 255;
        let Cx = Math.max( r, g, b );
        let Cn = Math.min( r, g, b );
        let d = Cx - Cn;
        let lev = (Cx + Cn) / 2;
        let sat = d / ( 1 - Math.abs( 2*lev - 1 ) );
        let hue;
        if ( d < tol ) {
            hue = 0;
            sat = 0;
        } else if ( Math.abs( Cx - r ) < tol ) {
            hue = ( (g - b) / d + 6 ) % 6; /* normalize potential negative */
        } else if ( Math.abs( Cx - g ) < tol ) {
            hue = ( (b - r) / d ) + 2;
        } else if ( Math.abs( Cx - b ) < tol ) {
            hue = ( (r - g) / d ) + 4;
        }
        return [ hue * 60, sat, lev ];
    }

};
