/**
* Copyright (c) 2021-2025 Kedron Holdings LLC, All Rights Reserved.
* This file is not part of Reactor, but a free add-on for it. This file only is offered under the MIT License.
* See https://mit-license.org for license text/details.
*/

const version = 25029;

const Controller = require("server/lib/Controller").requires( 24287 );
const Capabilities = require("server/lib/Capabilities");

const Logger = require("server/lib/Logger");
const util = require("server/lib/util");

Logger.getLogger( 'NUTController', Logger.getLogger( 'Controller' ) )
    .always( "Module NUTController v%1", version );

const NUT_PORT = 3493;

const STATUS_SET = {  // Ref: https://networkupstools.org/docs/developer-guide.chunked/new-drivers.html#_status_data
    OL: "On line",
    OB: "On battery",
    LB: "Low battery",
    HB: "High battery",
    RB: "Replace battery",
    CHRG: "Battery charging",
    DISCHRG: "Battery discharged",
    BYPASS: "Bypassed",
    CAL: "Calibrating",
    OFF: "Off line",
    OVER: "Overload",
    TRIM: "Trim/buck",
    BOOST: "Boost",
    FSD: "Forced Shutdown"
};

module.exports = class NUTController extends Controller {
    constructor( struct, id, config ) {
        super( struct, id, config );

        this.num_fail = 0;
        this.stopping = false;

        this.server = util.coalesce( this.config.server, "localhost" );
        this.port = util.coalesce( parseInt( this.config.port ), NUT_PORT );

        this.nut = false;
        this.nut_connected = false;
        this.first_query = true;

        if ( ! Capabilities.getCapability( 'x_nut_ups' ) ) {
            Capabilities.loadCapabilityData( {
                "x_nut_ups": {
                    "attributes": {},
                    "actions": {}
                }
            });
        }
    }

    /** Start the controller.
     */
    async start() {
        this.stopping = false;
        this.startDelay( 50 );
        return this;
    }

    async stop() {
        this.stopping = true;

        if ( this.nut ) {
            try {
                this.nut.close();
            } catch ( err ) {
                this.log.exception( err );
            } finally {
                this.nut = false;
            }
        }
        this.nut_connected = false;

        await super.stop();
    }

    run() {
        const self = this;

        if ( ! this.nut ) {
            this.start_client();
            this.startDelay( 1000 );
            return;
        }
        if ( ! this.nut_connected ) {
            this.log.info( "%1 client not yet ready; waiting", this );
            this.startDelay( 1000 );
            return;
        }

        this.nut.GetUPSList( async ( upslist, err ) => {
            try {
                if ( null === err && null !== upslist ) {
                    // Successful query
                    const keylist = Object.keys( upslist );
                    if ( self.first_query ) {
                        self.log.info( "%1 initial query succeeded with %2 items returned", self, keylist.length );
                        self.online();
                        self.num_fail = 0;
                    }
                    self.log.debug( 5, "%1 querying upslist=%2", self, keylist );
                    const allEntities = new Set( Object.keys( self.getEntities() ) );
                    allEntities.delete( "system" );
                    allEntities.delete( "controller_all" );

                    for ( const upsname of keylist ) {
                        self.log.debug( 5, "%1 getting vars for %2", self, upsname );

                        const e = self.getEntity( 'Entity', upsname );
                        e.deferNotifies( true );

                        try {
                            const vars = await( self._get_ups_vars( upsname ) );


                            // Ref: https://networkupstools.org/docs/developer-guide.chunked/ar01s09.html#_type
                            // Save all variables in extended attributes
                            // Map to capabilities: power_source, battery_maintenance, battery_power
                            // Ref: https://networkupstools.org/docs/developer-guide.chunked/apas02.html
                            // Interesting, perhaps:
                            // ups.status, ups.alarm, ups.model, ups.mfr, ups.serial, ups.temperature, ups.load, ups.type, ups.realpower,
                            // input.voltage, input.voltage.status, input.current, input.frequency, input.load, input.power, input.source
                            // output.voltage, output.current, battery.charge, battery.voltage, battery.charger.status, battery.status,
                            // battery.date.maintenance
                            if ( vars[ 'device.type' ] !== 'ups' ) {
                                self.log.debug( 5, '%1 ignoring unsupported device %2 type %3', self, upsname, vars['device.type'] );
                                continue;
                            }

                            if ( self.log.getEffectiveLevel() >= 6 ) {
                                Object.keys( vars ).forEach( v => {
                                    self.log.debug( 6, "%1 %2=%3", self, v, vars[v] );
                                });
                            }

                            const stat = util.coalesce( vars[ 'ups.status' ], "Unknown" ).split( /\s+/ );

                            let batt_replace = null;
                            if ( vars[ 'battery.date.maintenance' ] ) {
                                try {
                                    const when = new Date( vars[ 'battery.date.maintenance' ] );
                                    batt_replace = Date.now() >= when;
                                } catch ( err ) {
                                    self.log.notice( "%1 %2 can't interpret next maintenance date %3: %4", self,
                                        upsname, vars[ 'battery.date.maintenance' ], err );
                                }
                            }

                            if ( self.first_query || ! allEntities.has( upsname ) ) {
                                // Initial query or new entity extends what we need, refreshes
                                self.log.info("%1 initializing %2", self, upsname );
                                e.extendCapability( 'battery_maintenance' );
                                e.extendCapability( 'battery_power' );
                                e.extendCapability( 'power_source' );
                                e.extendCapability( 'x_nut_ups' );
                                e.refreshCapabilities();
                            }

                            e.setAttributes( {
                                rechargeable: true,
                                replace: batt_replace,
                                charging: vars['battery.charger.status'] === 'charging' || stat.includes( "CHRG" )
                            }, 'battery_maintenance' );

                            const charge = util.coalesce( parseInt( vars['battery.charge'] ) / 100 );
                            if ( e.getAttribute( 'battery_power.level' ) !== charge ) {
                                e.setAttributes( {
                                    level: charge,
                                    since: Date.now()
                                }, 'battery_power' );
                            }

                            e.setAttribute( 'power_source.source',
                                stat.includes( "OFF" ) ? "offline" : ( stat.includes( "OB" ) ?  "battery" : "utility" )
                            );

                            // User-readable status
                            const m = stat.map( el => STATUS_SET[ el ] || el );
                            e.setAttribute( "x_nut_ups.user_status_message", m.join( "; " ) );
                            e.setPrimaryAttribute( "x_nut_ups.user_status_message" );

                            // All response variables exported as extended attributes
                            Object.keys( vars ).forEach( v => {
                                e.setAttribute( `x_nut_ups.${v.replace( ".", "_" )}`, util.coalesce( vars[ v ] ) );
                            });
                        } catch ( err ) {
                            self.log.err( "%1 failed to get detail for %2: %3", self, upsname, err );
                            if ( err.message.match( /DATA-STALE/i ) ) {
                                self.log.notice( "%1 NUT is reporting a UPS communication problem. Please see the articles below for information. Also consult your operating system logs and the NUT documentation.",
                                    self );
                                self.log.notice( "https://raspberrypi.stackexchange.com/questions/66611/nut-cyberpower-data-stale" );
                                self.log.notice( "https://networkupstools.org/docs/man/upsd.conf.html" );
                                e.setAttributes({
                                    "battery_power.level": null,
                                    "battery_power.since": Date.now(),
                                    "power_source.source": null,
                                    "battery_maintenance.charging": null,
                                    "x_nut_ups.user_status_message": "UPS communication error"
                                });
                            } else {
                                self.log.exception( err );
                            }
                        } finally {
                            e.markDead( false );
                            e.deferNotifies( false );
                            allEntities.delete( upsname );
                        }
                    }

                    // Anything we didn't get? Mark it dead.
                    for ( const upsname of allEntities ) {
                        try {
                            const e = self.getEntity( 'Entity', upsname );
                            e.markDead( true );
                        } catch ( err ) {
                            self.log.err( "%1 marking %2 dead: %3", self, upsname, err );
                        }
                    }

                    // Mark first query completed.
                    if ( self.first_query ) {
                        self.purgeDeadEntities();
                    }
                    self.log.debug( 5, "%1 marking first query done!", self );
                    self.first_query = false;
                } else {
                    // Failed inventory query
                    self.log.err( "%1 can't get UPS list: %2", self, err );
                    self.recycle();
                }
            } catch ( err ) {
                self.log.exception( err );
            } finally {
                self.startDelay( util.coalesce( parseInt( this.config.poll_interval ), 10 ) * 1000 );
            }
        });
    }

    // Wrapper for callback-based function
    async _get_ups_vars( upsname ) {
        const self = this;
        return new Promise( (resolve,reject) => {
            self.nut.GetUPSVars( upsname, ( vars, err ) => {
                if ( err ) {
                    reject( new Error( err ) );
                } else {
                    resolve( vars );
                }
            });
        });
    }

    recycle() {
        if ( this.nut ) {
            try {
                this.log.debug( 5, "%1 forcing connection closed for recycle", this );
                this.nut.removeAllListeners( "close" );
                this.nut.removeAllListeners( "error" );
                this.nut.close();
            } catch ( err ) {
                /* nada */
            } finally {
                this.nut = false;
            }
        }
        this.nut_connected = false;

        /* Arm for reconnect */
        this.stopTimer();
        if ( ! this.stopping ) {
            const dly = Math.min( 60000, Math.max( 2000, ( this.num_fail - 30 ) * 10000 ) );  /* Retry decay */
            this.log.debug( 4, "%1 recycling/reconnecting in %2ms (%3 fails)", this, dly, this.num_fail );
            this.startDelay( dly );
        }
    }

    start_client() {
        const self = this;
        const Nut = require( 'node-nut' );  // Ref: https://github.com/skarcha/node-nut

        if ( this.nut ) {
            try {
                this.nut.close();
            } catch ( err ) {
                /* nada */
            } finally {
                this.nut = false;
            }
        }

        this.nut_connected = false;
        this.first_query = true;
        this.nut = new Nut( this.port, this.server );

        this.nut.on( 'error', err => {
            self.stopTimer();
            if ( ! self.nut_connected ) {
                /* Error during initial connection */
                self.log.err( "%1 unable to establish communication with %2:%3: %4", self, self.server, self.port, err );
                if ( ++self.num_fail >= 3 ) {
                    self.offline();
                }
            } else {
                self.log.err( "%1 lost communication due to error: %2", self, err );
            }
            self.recycle();
        });

        this.nut.on( 'close', () => {
            self.log.notice( "%1 connection closed; attempting reconnect...", self );
            self.nut_connected = false;
            self.nut = false;
            self.stopTimer();
            if ( ! self.stopping ) {
                self.startDelay( 50 );
            }
        });

        this.nut.on( 'ready', () => {
            self.log.info( "%1 connected to %2:%3", self, self.server, self.port );
            self.stopTimer();
            if ( "" !== ( self.config.username || "" ) ) {
                self.log.info( "%1 setting client username (%2) and password", self, self.config.username );
                self.nut.SetUsername( self.config.username, ( err ) => {
                    if ( err ) {
                        self.log.err( "%1 failed to set username: %2", self, err );
                        if ( ++self.num_fail >= 3 ) {
                            self.offline();
                        }
                        self.recycle();
                    } else {
                        self.nut.SetPassword( self.config.password || "", ( err ) => {
                            if ( err ) {
                                self.log.err( "%1 failed to set password: %2", self, err );
                                if ( ++self.num_fail >= 3 ) {
                                    self.offline();
                                }
                                self.recycle();
                            } else {
                                self.nut_connected = true;
                                self.log.info( "%1 client ready after authentication", self );
                                // num_fail is set and online() called when we get a valid UPS list response
                            }
                        });
                    }
                });
            } else {
                self.nut_connected = true;
                self.log.info( "%1 client ready", self );
                // num_fail is set and online() called when we get a valid UPS list response
            }
            self.startDelay( 50 );
        });

        this.log.notice( "%1 starting NUT client with %2:%3; waiting for ready...", this, this.server, this.port );
        this.nut.start();
    }
};
