/**
* Copyright (c) 2021-2025 Kedron Holdings LLC, All Rights Reserved.
* This file is not part of Reactor, but a free add-on for it. This file only is offered under the MIT License.
* See https://mit-license.org for license text/details.
*/

const version = 25253;

const Logger = require("server/lib/Logger");
const Controller = require("server/lib/Controller").requires( 22125 );
const Entity = require("server/lib/Entity").requires( 25243 );
const { coalesce } = require( "server/lib/util" );

const DEFAULT_BOOT_INTERVAL = 5000;
const DEFAULT_POLL_INTERVAL = 60000;
const DEFAULT_TEMP_INTERVAL = 60000;

Logger.getLogger( 'OctoPrintController', Logger.getLogger( 'Controller' ) )
    .always( "Module OctoPrintController v%1", version );

module.exports = class OctoPrintController extends Controller {
    static createControllerGroup = false;

    constructor( struct, id, config ) {
        super( struct, id, config );

        this.num_fail = 0;
        this.stopping = true;
        this.control_entity = false;
        this.control_enabled = true;
        this.booting = false;

        /* Since this Controller only represents one thing, use the system entity to carry it. */
        this.removeControllerGroup();  // remove later ??? deprecated in Controller 25253
        this.system.extendCapability( 'printer3d' );

        this.interval = Math.max( DEFAULT_BOOT_INTERVAL, coalesce( parseInt( this.config.interval ), DEFAULT_POLL_INTERVAL ) );
        this.temp_interval = Math.max( 60000, coalesce( parseInt( this.config.temperature_interval ), DEFAULT_TEMP_INTERVAL ) );
        this.temp_hysteresis = Math.max( 0.1, coalesce( parseFloat( this.config.temperature_hysteresis ), 1.0 ) );
    }

    /** Start the controller. */
    async start() {
        this.stopping = false;
        this.sockjs_auth = false;
        this.sockjs_starting = false;

        if ( this.config.control_entity ) {
            let e = this.getStructure().findEntity( this.config.control_entity );
            if ( !e ) {
                this.log.err( "%1 control entity %2 not found", this, this.config.control_entity );
                this.sendError( "The configured control entity {0:q} was not found",
                    this.config.control_entity );
            } else if ( ! e.hasCapability( 'power_switch' ) ) {
                this.log.err( "%1 control entity %2 does not have power_switch capability", this, this.config.control_entity );
                this.sendError( "The configured control entity {0:q} does not provide the power_switch capability.",
                    this.config.control_entity );
            } else {
                this.control_entity = e;
                this.subscribe( e );
                this.control_enabled = e.getAttribute( 'power_switch.state' ) !== false;  /* null OK */
                this.log.info( "%1 control entity state: %2", this, this.control_enabled );
            }
        }
        this.run();
        this.online();  /* unconditionally */
        return this;
    }

    async stop() {
        this.stopping = true;

        if ( this.sockjs_sock ) {
            this.sockjs_sock.close();
        }

        return await super.stop();
    }

    notify( event ) {
        try {
            this.log.debug( 6, "%1 event notification %2", this, event );
            if ( this.control_entity && event.sender === this.control_entity ) {
                const newstate = this.control_entity.getAttribute( 'power_switch.state' ) !== false;  /* null OK */
                if ( newstate !== this.control_enabled ) {
                    this.log.info( "%1 control entity %2 state change to %3", this, this.control_entity, newstate );
                    this.control_enabled = newstate;
                    if ( newstate ) {
                        this.booting = true;
                    }
                    this._reset_timer( 10 );  // quickly... figure out what's going on!
                }
            } else if ( event.sender !== this ) {
                this.log.debug( 5, "%1 unhandled notification %2", this, event );
            }
        } catch ( err ) {
            this.log.err( "%1 error handling event %2 from %3: %4", this, event.type, event.sender, err );
            this.log.exception( err );
        } finally {
            super.notify( event );  /* MANDATORY */
        }
    }

    run() {
        this.stopTimer();

        const e = this.system;
        if ( this.control_entity && ! this.control_enabled ) {
            this.log.notice( "%1 printer control device %2 is off; going idle.", this, this.control_entity );
            this._printer_offline( false, "Control power off" );
            this.num_fail = 0;
            this.booting = false;
            // No new delay/timer. We wait for switch on notification.
            return;
        }

        // Attempt sockjs connection
        // Ref: https://docs.octoprint.org/en/master/api/push.html
        if ( ! this.sockjs_sock && ! this.sockjs_starting && false !== this.config.use_sockjs ) {
            const SockJS = require( 'sockjs-client' );
            const self = this;
            this.sockjs_starting = true;

            // Passive login.
            if ( ! this.sockjs_auth ) {
                this.log.info( "%1 attemping auth (using API key) with %2", this, this.config.source );
                this.fetchJSON( `${this.config.source}/api/login?passive=true&apikey=${ this.config.api_key }`,
                    {
                        method: "POST",
                        timeout: this.config.timeout || 15000
                    }
                ).then( data => {
                    self.log.debug( 5, "%1 received auth reply %2", self, JSON.stringify( data, null, 4 ) );
                    self.sockjs_auth = JSON.stringify( { "auth": `${data.name}:${data.session}` } );
                    self.log.debug( 5, "%1 auth is %2", self, self.sockjs_auth );
                    self.sockjs_starting = false;
                    self.booting = false;  // we know printer is up
                    self._reset_timer( 10 );  // fast turn-around to try SockJS
                }).catch( err => {
                    self.log.err( "%1 auth error: %2", self, err );
                    self.sockjs_starting = false;
                    self.sockjs_sock = false;
                    self.sockjs_auth = false;
                    if ( self.booting ) {
                        self._reset_timer( DEFAULT_BOOT_INTERVAL );
                    } else {
                        self._reset_timer( DEFAULT_POLL_INTERVAL );
                    }
                });

                // Default interval delay.
                this._reset_timer();
                return;
            }

            // Establish the SockJS connection
            let url = `${this.config.source}${this.config.sockjs_path || "/sockjs"}`;
            //url = url.replace( /^http/, "ws" );
            this.log.debug( 5, "%1 attempting SockJS connection to %2", this, url );
            const sock = new SockJS( url, undefined, { timeout: 10000 } );
            sock.onopen = this._sockjs_open.bind( this, sock );
            sock.onmessage = this._sockjs_message.bind( this );
            sock.onclose = this._sockjs_close.bind( this );

            /** Start a 60-second delay to the next run. If no message is received over the SockJS channel,
             *  this delay will end with a poll/query of the printer for update. This ensures that the
             *  stability and performance of the SockJS connection, which is relatively untested here,
             *  doesn't result in a no-update condition of the entity(ies).
             */
            this.startDelay( this.interval );
            return;
        }

        /** If we get here, the SockJS connection is not open (and apparently can't open), or no message
         *  was received there for a minute or more. Poll the printer.
         */
        this.log.debug( 5, "%1 querying printer at %2", this, this.config.source );
        this.fetchJSON( this.config.source + '/api/printer',
            {
                timeout: this.config.timeout || 15000,
                headers: { 'x-api-key': this.config.api_key }
            }
        ).then( data => {
            this.num_fail = 0;
            this.booting = false;
            this.log.debug( 5, "%1: received response data %2", this, data );
            e.deferNotifies( true );
            e.markDead( false );
            try {
                e.setAttribute( 'printer3d.reachable', true );
                e.setAttribute( 'printer3d.printing', coalesce( data.state?.flags?.printing ) );
                e.setAttribute( 'printer3d.paused', coalesce( data.state?.flags?.paused ) );
                e.setAttribute( 'printer3d.resuming', coalesce( data.state?.flags?.resuming ) );
                e.setAttribute( 'printer3d.ready', coalesce( data.state?.flags?.ready ) );
                e.setAttribute( 'printer3d.operational', coalesce( data.state?.flags?.operational ) );
                e.setAttribute( 'printer3d.error', coalesce( data.state?.flags?.error ) );
                e.setAttribute( 'printer3d.finishing', coalesce( data.state?.flags?.finishing ) );
                e.setAttribute( 'printer3d.temperature_bed_actual', coalesce( data.temperature?.bed?.actual ) );
                e.setAttribute( 'printer3d.temperature_bed_target', coalesce( data.temperature?.bed?.target ) );
                e.setAttribute( 'printer3d.temperature_hotend_actual', coalesce( data.temperature?.tool0?.actual ) );
                e.setAttribute( 'printer3d.temperature_hotend_target', coalesce( data.temperature?.tool0?.target ) );
                e.setAttribute( 'printer3d.message', coalesce( data.state?.text ) );
                e.setAttribute( 'printer3d.sdcard_ready', coalesce( data.sd?.ready ) );
            } catch ( err ) {
                this.log.exception( err );
            } finally {
                e.deferNotifies( false );
                this._reset_timer();
            }
        }).catch( err => {
            this.log.err( "%1: error fetching printer state: %2", this, err );
            if ( this.booting ) {
                this.startDelay( this.config.boot_interval || DEFAULT_POLL_INTERVAL );
                this._printer_offline( false, "Control power on; waiting for printer ready" );
            } else {
                this.startDelay( Math.min( 120000, ( this.config.error_interval || DEFAULT_POLL_INTERVAL ) * Math.max( 1, ++this.num_fail - 12 ) ) );
                if ( this.num_fail >= 3 ) {
                    this._printer_offline( true, `${this.config.source} ${String(err)}` );
                }
            }
        });
    }

    _printer_offline( isError, msg ) {
        const e = this.system;
        this.log.debug( 5, "%1 setting printer offline, error=%2: %3", this, !! isError, msg );
        e.setAttributes({
            reachable: false,
            printing: false,
            paused: false,
            resuming: false,
            ready: false,
            operational: false,
            error: !! isError,
            finishing: false,
            temperature_bed_actual: null,
            temperature_bed_target: null,
            temperature_hotend_actual:  null,
            temperature_hotend_target: null,
            message: msg || null,
            sdcard_ready: null,
            jobname: null,
            currentz: null
        }, 'printer3d' );
        e.deferNotifies( false );
    }

    _reset_timer( delay ) {
        this.stopTimer();
        this.startDelay( delay || this.interval );
    }

    _sockjs_open( sock ) {
        this.sockjs_sock = sock;
        this.log.debug( 5, "%1 sockjs socket open; sending auth and subscription request", this );
        this.sockjs_sock.send( JSON.stringify( { "subscribe": { "state": true, "events": true } } ) );
        this.sockjs_sock.send( this.sockjs_auth );
        this.sockjs_sock.send( JSON.stringify( { "throttle": Math.max( 1, coalesce( parseInt( this.config.throttle ), 4 ) ) } ) );
        this.log.notice( "%1 SockJS connection to printer established!", this );
        this._reset_timer();
    }

    // Ref: https://docs.octoprint.org/en/master/api/push.html
    _sockjs_message( event ) {
        this.log.debug( 6, "%1 SockJS socket received %2", this, event.type );
        this.log.debug( 6, "%1 full data %2", this, JSON.stringify( event, null, 4 ) );
        if ( "message" === event.type && event.data.reauthRequired ) {
            this.log.info( "%1 printer requires re-auth; recycling connection", this );
            this.sockjs_auth = false;
            this.sockjs_sock.close();  // close handler will reset timer
            return;
        } else if ( "message" === event.type && event.data.connected ) {
            const d = event.data.connected;
            this.log.notice( "%1 connected to OctoPrint %2 online %3 safe_mode %4",
                this, d.display_version, d.online, d.safe_mode );
            //this.log.raw(JSON.stringify(event,null,4));
        } else if ( "message" === event.type && ( event.data?.current || event.data?.history ) ) {
            const now = Date.now();
            const current = event.data.current || event.data?.history;
            const e = this.getSystemEntity();
            // current.state, current.job, current.progress, current.currentZ, current.temps
            try {
                e.deferNotifies( true );
                if ( current.state ) {
                    const state = current.state;
                    this.log.debug( 5, "%1 SockJS state update %2", this, state );
                    e.setAttributes( {
                        "reachable": true,
                        "printing": coalesce( state.flags?.printing ),
                        'paused': coalesce( state.flags?.paused ),
                        'resuming': coalesce( state.flags?.resuming ),
                        'ready': coalesce( state.flags?.ready ),
                        'operational': coalesce( state.flags?.operational ),
                        'error': coalesce( state.flags?.error ),
                        'finishing': coalesce( state.flags?.finishing ),
                        'message': coalesce( state.text )
                    }, "printer3d" );
                }
                if ( current.job ) {
                    this.log.debug( 5, "%1 SockJS job %2", this, current.job );
                    e.setAttribute( 'printer3d.jobname', coalesce( current.job.file?.display ) );
                }
                if ( current.progress ) {
                    this.log.debug( 5, "%1 SockJS progress %2", this, current.progress );
                }
                if ( current.currentZ ) {
                    e.setAttribute( 'printer3d.currentz', coalesce( current.currentZ ) );
                }

                /** Do temps last, because we may apply hysteresis if no other changes have been made to the entity
                 *  by the current states.
                 *
                 *  Temperature updates come fast, so this could update the entity a lot. To avoid spamming our-
                 *  selves with entity changes, we only post temperature changes under the following conditions:
                 *      1. Any other attribute of the entity has changed (i.e. the entity is "dirty");
                 *      2. The difference between the last temperature and the current is more than the configured
                 *         hysteresis value (by temp_hysteresis; current default is 1.0 degrees);
                 *      3. It has been a minute or more since temperature was updated.
                 */
                if ( Array.isArray( current.temps ) && current.temps.length > 0 ) {
                    const temp = current.temps[0];
                    this.log.debug( 5, "%1 SockJS temperature update %2", this, temp );
                    // This updates the entity a lot, because of the 1/10th degree resolution of temps.
                    // Idea... if target temps are 0, only update every 60 seconds or 1 degree
                    // May need to do something similar while printing, too.
                    e.setAttribute( 'printer3d.temperature_bed_target', coalesce( temp.bed?.target ) );
                    e.setAttribute( 'printer3d.temperature_hotend_target', coalesce( temp.tool0?.target ) );
                    let changed = e.isDirty();  // true of entity has been modified by any of the above.
//this.log.always("%1 temperature update changed=%2 going in", this,changed);
                    // Check bed temperature (if necessary).
                    const bta = coalesce( temp.bed.actual );
                    if ( ! changed ) {
                        const lbta = coalesce( e.getAttribute( 'printer3d.temperature_bed_actual' ), 0 );
//this.log.always("%1 bed temp %2 was %3 diff %4 age %5", this, bta, lbta, lbta - bta, ( now - e.getAttributeMeta( 'printer3d.temperature_bed_actual' )['last_modified'] ) );
                        changed = changed ||
                            null === lbta ||
                            Math.abs( lbta - bta ) >= this.temp_hysteresis ||
                            ( now - e.getAttributeMeta( 'printer3d.temperature_bed_actual' )['last_modified'] ) >= this.temp_interval;
                    }

                    // Check extruder temperature (if necessary).
                    const tta = coalesce( temp.tool0?.actual );
                    if ( ! changed ) {
                        const ltta = coalesce( e.getAttribute( 'printer3d.temperature_hotend_actual' ), 0 );
//this.log.always("%1 tool temp %2 was %3 diff %4 age %5", this, tta, ltta, ltta - tta, ( now - e.getAttributeMeta( 'printer3d.temperature_hotend_actual' )['last_modified'] ) );
                        changed = changed ||
                            null === ltta ||
                            Math.abs( ltta - tta ) >= this.temp_hysteresis ||
                            ( now - e.getAttributeMeta( 'printer3d.temperature_hotend_actual' )['last_modified'] ) >= this.temp_interval;
                    }

                    if ( changed ) {
//this.log.always("%1 POST TEMP CHANGE", this);
                        e.setAttributes( {
                            'temperature_bed_actual': bta,
                            'temperature_hotend_actual': tta
                        }, "printer3d" );
                    }
                }
            } finally {
                e.deferNotifies( false );

                this._reset_timer();
            }
        } else {
            this.log.debug( 5, "%1 unhandled message: %2 with %3", this, event.type, Object.keys( event.data || {} ) );
            if ( this.log.getEffectiveLevel() >= 5 ) {
                this.log.raw( JSON.stringify( event, null, 4 ) );
            }
        }
    }

    _sockjs_close( e ) {
        this.log.notice( "%1 SockJS connection closed (%2)", this, e.code );
        //console.log("sockjs socket closed",e);
        this.sockjs_sock = false;
        this.sockjs_starting = false;
        this.stopTimer();
        if ( ! this.stopping ) {
            if ( e.code === 1002 ) {
                this.booting = true;
                this.sockjs_auth = false;
                this.startDelay( this.boot_interval );
            } else {
                this.startDelay( 10 );  // fast reconnect
            }
        } else {
            this._printer_offline( false, `${this.getClass()}#${this.getID()} stopping` );
        }
    }
};
