/** An example Reactor controller for the WhiteBIT WebSocket API.
 *  Copyright (c) 2022 Patrick H. Rigney, All Rights Reserved.
 *  WhiteBitController is offered under MIT License - https://mit-license.org/
 *  More info: https://reactor.toggledbits.com/
 *
 *  Disclaimer: My selection of this API for this example is strictly
 *  matter of convenience, as the API is free to use and well-documented.
 *  This is in no way an endorsement of WhiteBIT, nor have I received any
 *  consideration for the free advertising I am providing them here. :)
 */

/** For a description of the WhiteBIT WebSocket API, please see:
 *  https://whitebit-exchange.github.io/api-docs/docs/Public/websocket
 */

/** NOTE! This is really just example code, and while it is a fully working
 *  Controller subclass for Reactor, in practice this API sends a LOT of messages
 *  and would make your system crazy busy if you started writing conditions
 *  against these entities. But for that, it's a great demo because you can
 *  really see a lot of action. Here's a sample `controllers` entry to get you
 *  going:
 *
 *  - id: whitebit
 *    name: WhiteBIT
 *    implementation: WhiteBITController
 *    enabled: true
 *    config:
 *      markets:
 *        - ETH_BTC
 *        - BTC_USD
 */

const version = 22305;

const Controller = require("server/lib/Controller");

const Logger = require( "server/lib/Logger" );
Logger.getLogger( 'WhiteBITController', 'Controller' ).always( "Module WhiteBITController v%1", version );

const Configuration = require("server/lib/Configuration");
const logsdir = Configuration.getConfig( "reactor.logsdir" );  /* logs directory path if you need it */

/* Another other modules we need? */
const util = require("server/lib/util");

const ERROR_DELAY = 5000;

var impl = false;  /* Implementation data, one copy for all instances, will be loaded by start() later */

module.exports = class WhiteBITController extends Controller {
    constructor( struct, id, config ) {
        super( struct, id, config );  /* required */

        this.stopping = true;       /* Flag indicates we're stopping */

        this.next_req_id = 1;       /* Next request ID for tracked requests */
        this.pending = new Map();   /* Pending requests (keys are integer, values are TimedPromise) */
    }

    /** Start the controller.
     */
    async start() {
        /* Make sure our configuration gives us data to query */
        if ( 0 === ( this.config.markets || [] ).length ) {
            return Promise.reject( "No markets configured" );
        }

        /** Load implementation data if not yet loaded. Remove this if you don't
         *  use implementation data files.
         */
        if ( false === impl ) {
            /*  The string passed below should be the **lowercase** translation
             *  of your class basename.
             */
            impl = await this.loadBaseImplementationData( 'whitebit', __dirname );
        }

        /** Do what you need to do to start your controller here. How you
         *  structure this code is up to you. Here, for example purposes,
         *  we'll just call our run() method and kick of a train of timer
         *  events.
         */
        this.stopping = false;
        this.run();

        /** Must return `this` or a Promise that resolves to this. If this
         *  is declared async (as it should be), the return of a Promise
         *  is implicit (async functions always return Promise).
         */
        return this;
    }

    /** Stop the controller.
     *  You only need to supply this method if your subclass has something
     *  specific it needs to do.
     */
    async stop() {
        this.log.notice( "%1 stopping", this );
        this.stopping = true;

        /* Required ending */
        return await super.stop();
    }

    /** run() is called when Controller's single-simple timer expires.
     *  If you don't use Controller's simple timer, you don't need this.
     */
    run() {
        if ( ! this.ws_connected() ) {
            this.connect_whitebit();
            return;
        }
    }

    /** performOnEntity() - we don't have any actions, we're just a data mining
     *  tool (read only), so we don't override the superclass method and let it
     *  handle anything that comes up (which will be nothing).
     */
    /* performOnEntity( entity, actionName, parameters ) { } */

    /** ---------------- ALL OF YOUR CODE BELOW ------------------ */

    /** connect_whitebit() opens the WebSocket connection to the WhiteBIT API. It then
     *  requests an initial set of prices for the markets we care about, and requests
     *  a subscription to price changes for those markets. If all that works, it signals
     *  that this controller is on-line. The API requires a ping every 60 seconds, so we
     *  set our pingInterval to 45 just to be safe.
     */
    connect_whitebit() {
        const url = `${this.config.url || "https://api.whitebit.com"}/ws`;
        this.log.info( "%1 connecting to %2", this, url );
        this.ws_open( url, { pingInterval: 45000 } ).then( async () => {
            const markets = this.config.markets || [];
            this.log.info( "%1 websocket API connected, querying last prices on %2 markets",
                this, markets.length );
            for ( let market of markets ) {
                let response = await this.send_tracked_request( 'lastprice_request', [ market ] );
                this.initialize_entity( market, response );
            }

            /* Now subscribe to updates from these markets */
            this.send_tracked_request( 'lastprice_subscribe', markets ).then( result => {
                this.log.debug( 5, "%1 subscribe result %2", this, result );
                /* Setups are done. Signal online. */
                this.online();
            }).catch( err => {
                this.log.err( "%1 failed to subscribe to markets: %2", this, err );
                throw err;
            });
        }).catch( err => {
            this.log.err( "%1 error connecting to API: %2", this, err );
            this.ws_terminate();  /* Make sure connection is closed and gone */
            this.startDelay( ERROR_DELAY );
        });
    }

    /** send_tracked_request - WhiteBIT WebSocket API uses a JSON-RPC model, where we send
     *  a request with a unique ID, and we later get back an answer with that ID to tie it
     *  together. This method sends a tracked request and returns a TimedPromise for it.
     *  The request will resolve with the matching response, or reject with an Error or the
     *  string "timeout".
     */
    send_tracked_request( method, params, timeout ) {
        timeout = timeout || 15000;
        let slot = {
            id: this.next_req_id++,
            method: method,
            expires: Date.now() + timeout,
            resolve: false,
            reject: false
        };

        /* Build the request payload */
        let req = { id: slot.id, method: method, params: params };

        /** Wrap the send in a TimedPromise. If the response comes back before the timeout,
         *  the TimedPromise resolves with the response data (see ws_message). Otherwise,
         *  it will reject with some error or the string "timeout".
         */
        const self = this;
        slot.promise = util.TimedPromise( (resolve,reject) => {
            self.log.debug( 6, "%1 running TimedPromise for %2", self, slot.method );
            /** Save the Promises resolve and reject functions for later use when the response
             *  comes (see ws_message ).
             */
            slot.resolve = resolve;
            slot.reject = reject;

            /* Send the payload */
            self.log.debug( 6, "%1 sending tracked request %2: %3", this, slot.id, req );
            self.ws_send( JSON.stringify( req ) );
        }, timeout ).catch( err => {
            self.log.err("%1 request %2 (%4) failed: %3", self, slot.id, err, slot.method);
            throw new Error( "request failed" );
        }).finally( () => {
            self.log.debug( 6, "%1 removing settled tracked request %2", self, slot.id );
            self.pending.delete( slot.id );
        });

        /* Add the slot to our map */
        this.pending.set( slot.id, slot );

        /* Return its TimedPromise */
        this.log.debug( 6, "%1 created tracked request %2 for %3", this, slot.id, slot.method );
        return slot.promise;
    }

    /** clear_tracked_requests() clears (rejects) all existing tracked requests. This is usually
     *  done in response to stopping or loss of connection.
     */
    clear_tracked_requests() {
        for ( let [ id, slot ] in this.pending.entries() ) {
            try {
                slot.reject( "connection closed" );
            } catch ( err ) {
                this.log.warn( "%1 while terminating tracked request %2: %3", this, id, err );
            }
        }
    }

    /** This method is called when the websocket signals that it's closing. We start a
     *  timer that will execute the run() method when it expires to reconnect.
     */
    ws_closing( code, reason ) {
        this.log.notice( "%1 websocket closing!" );
        this.clear_tracked_requests();

        if ( ! this.stopping ) {
            this.log.notice( "%1 arming to reconnect..." );
            this.stopTimer();  /* Whatever it was doing, not important now */
            this.startDelay( ERROR_DELAY );
        }

        return super.ws_closing( code, reason );
    }

    /** Handle inbound WebStocket datagrams. These will be either responses for tracked requests,
     *  or price update notifications from our subscription.
     */
    ws_message( buf ) {
        this.log.debug( 6, "%1 handling message %2 (%3)", this, buf, typeof buf );
        let data = JSON.parse( buf );
        if ( data.id ) {
            /* Messages with IDs are tracked request responses. Find the tracked request slot. */
            let slot = this.pending.get( data.id );
            if ( ! slot ) {
                /* No slot for ID, may have timed out already, or not right msg type? */
                return;
            }
            /* Decide if we succeeded or failed. */
            if ( data.error ) {
                slot.reject( data.error );
            } else {
                slot.resolve( data.result );
            }
            /* The TimedPromise finally() will remove the slot from our Map */
        } else if ( "lastprice_update" === data.method ) {
            /* Price update message. */
            this.log.debug( 5, "%1 got price update %2", this, data );
            let e = this.findEntity( data.params[0] );
            if ( ! e ) {
                this.log.notice( "%1 received unexpected update for %2", this, data.params[0] );
                return;
            }
            this.update_entity( e, data.params[1] );
        } else {
            this.log.debug( 5, "%1 ignoring unknown message %2", this, data );
        }
    }

    /** Initialize the entity for this market with its price. This is called during
     *  startup (and on reconnects) to make sure the entity is properly set up. It's
     *  possible the entity already exists and was restored from persistent storage,
     *  so we need to consider both the case of creating new and updating from
     *  restored states.
     */
    initialize_entity( market, data ) {
        this.log.debug( 5, "%1 initialize_entity(%2, %3)", this, market, data );
        let e = this.findEntity( market );
        if ( ! e ) {
            /* New entity. Create it. */
            this.sendNotice( "Creating entity for new market {0}", market );
            e = this.getEntity( 'Entity', market );
            e.extendCapability( 'value_sensor' );
            e.setPrimaryAttribute( 'value_sensor.value' );
            e.setName( market );
        }

        try {
            e.markDead( false );
            this.update_entity( e, data );
        } catch ( err ) {
            this.log.err( "%1 failed to initialize %2: %3", this, market, err );
        } finally {
            e.deferNotifies( false );  /* Make sure it's released */
        }
    }

    /** Update our entity with new price data. */
    update_entity( e, data ) {
        /** Deferring notifies isn't really needed for this simple update of one
         *  attribute, but we do it anyway because this is example code, and if
         *  you were updating multiple attributes, this is exactly what you would do.
         */
        e.deferNotifies( true );
        try {
            e.setAttribute( 'value_sensor.value', parseFloat( data ) );
        } catch ( err ) {
            this.log.err( "%1 failed to update %2: %3" );
        } finally {
            e.deferNotifies( false );
        }
    }

    /** Default action handler for sys_system.restart. Just terminate our connection and
     *  let the close handler arm a reconnection.
     */
    action_sys_system_restart( entity, params ) {
        this.ws_close( 1002, "closing" );
        return Promise.resolve();
    }
};
