/* txtool.js - A tool to help translators find differences in translation files

   Usage: node tools/txtool.js <translation-file> [<reference-file>]

   NOTE: Must run from the install directory, not from inside "tools"!

   Compares the <translation-file> to the <reference-file>. If the
   <reference-file> is not given, the system's US English file is
   used.

   HOW TO USE THIS TOOL:

   Find New Strings You Need to Translate:
   This will compare your translation file to the installed version in the
   Reactor install directory (which presumably is from the "latest" release):

         node tools/txtool.js <path-to-your-file>

   Find Difference Between Reference Versions:
   If you give to filenames, both to US English reference versions, the older
   version first and the newer second, the tool will report new strings,
   removed strings, and changed values (that is, those strings where the key
   has remained the same but the string value has been modified).

         node tools/txtool.js -d <path-to-your-file>

   Find Difference Between Translation Versions:
   Give the tool two of your translation files, older first and newer second,
   and it will report the strings added, removed, and changed.

         node tools/txtool.js -d <path-to-your-file>

   Yes, you can also use "diff" to do all of this, but sometimes its output
   is less clear.
*/

const version = 23345;

const path = require( 'path' );
const fs = require( 'fs' );
const yaml = require('js-yaml');

function quot( s ) {
    if ( s.indexOf( "'" ) < 0 ) {
        return "'" + s + "'";
    }
    return '"' + s.replace( '"', "\\\"" ) + '"';
}

let detail=false;
let argv = process.argv.slice( 2 );
while ( argv.length > 0 ) {
    if ( ! argv[0].startsWith( '-' ) ) {
        break;
    }
    let arg = argv.shift();
    switch ( arg ) {
        case '-d':
            detail = true;
            break;
        default:
            console.error( "Invalid argument:", arg );
            process.exit(255);
    }
}
if ( 0 === argv.length ) {
    console.log( "Usage: node tools/txtool.js [-d] <translation-file> [<reference-file>]" );
    return 255;
}

let localeData, refData;

let infname = argv.shift();
try {
    let ff = fs.readFileSync( infname );
    localeData = yaml.load( ff, { filename: infname } );
} catch ( err ) {
    console.error( `Can't read ${infname}: ${err}` );
    console.error( err );
}
if ( "object" !== typeof localeData || "object" !== typeof localeData.strings ) {
    console.error( `${infname} does not appear to be a translation file` );
    process.exit( 1 );
}

let refname = argv.length > 0 ? argv.shift() : path.resolve( "./common", "locale_en-US.yaml" );
try {
    let ff = fs.readFileSync( refname );
    refData = yaml.load( ff, { filename: refname } );
} catch ( err ) {
    console.error( `Can't read ${refname}: ${err}` );
    console.error( err );
}

console.log( `Comparing ${localeData.locale} version ${localeData.version} to reference ${refData.locale} version ${refData.version}` );

let ltag = localeData.locale + ":" + localeData.version;
let rtag = refData.locale + ":" + refData.version;

let lkeys = Object.keys( localeData.strings );
let rkeys = Object.keys( refData.strings );

rkeys.forEach( k => {
    if ( "undefined" === typeof localeData.strings[k] ) {
        if ( k.startsWith( '#' ) ) {
            console.log( `${rtag}: added: ${quot(k)}: ${quot(refData.strings[k].replace( /[\r\n]+$/, "" ) )}` );
        } else {
            console.log( `${rtag}: added: ${quot(k)}` );
        }
    }
});

lkeys.forEach( k => {
    if ( "undefined" === typeof refData.strings[k] ) {
        console.log( `${rtag}: disused: ${quot(k)}` );
    } else if ( localeData.locale === refData.locale && refData.strings[k].replace( /[\r\n]+$/, "" ) != localeData.strings[k].replace( /[\r\n]+$/, "" ) ) {
        console.log( `${ltag}: value changed: ${quot(k)}` );
        if ( detail ) {
            console.log( `     was: ${JSON.stringify(localeData.strings[k])}` );
            console.log( `     now: ${JSON.stringify(refData.strings[k])}` );
        }
    }
});
